### A Pluto.jl notebook ###
# v0.20.21

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    #! format: off
    return quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
    #! format: on
end

# ╔═╡ de358d50-fa53-405c-85ad-b1f0b0ce3fbd
begin
	using HypertextLiteral, PlutoUI, ColorSchemes, PlutoPlotly, PlotlyKaleido
	using CSV, DataFrames, Dates, LinearAlgebra, NLsolve , StatsBase, Statistics
	using PlutoTeachingTools, ShiftedArrays, SparseArrays, CommonMark,PeriodicalDates
	PlotlyKaleido.start()
    force_pluto_mathjax_local(true)
    PlotlyKaleido.start(;mathjax = true)
end

# ╔═╡ 51e9ba0a-7793-4cf4-a0c1-bc90aecff9c6
html"""
<picture>
  <source srcset="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/negative/rgb_ibs_en_horizontal_negative.png" media="(prefers-color-scheme: dark)" width="240px!important" style="float:right;">
  <img src="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/main/rgb_ibs_en_horizontal_main.png" width="240px" style="float:right;">
</picture>
"""

# ╔═╡ 0125c893-46aa-457c-bd1b-18589e9ab2d3
md"""
# Day 1 - Welcome to Pluto Notebooks

## Introduction to Pluto
"""

# ╔═╡ 41a74197-ceb9-40b6-8d24-1712f8d93b1a
md"""
**Vivaldo Mendes**

**February 02, 2026**
"""

# ╔═╡ 2606832a-1d82-417a-93a8-623fe40149f6
md"""
---
"""

# ╔═╡ 182cbbb9-2783-4a46-865e-036df9ee1320
md"""
### Packages used in this course
"""

# ╔═╡ ded0a217-cd7b-4ba7-be70-9be19cfe8322
html"""
<style>
    kbd {
        background-color: #505050 !important;
        color: #fff !important;
        padding: 3px 6px;
        border-radius: 4px;
        font-family: monospace;
		font-size: 0.8rem !important;
        border: 1px solid #666;
    }
</style>
"""

# ╔═╡ 69c09e47-6c6e-441c-90e2-a3863d9ca71f
md"""
---
"""

# ╔═╡ 0d1f1f47-ecd8-4d16-80b4-d9154bc7cc9a
md"""
## 1. What is a Pluto notebook?
"""

# ╔═╡ ffadd7f9-ce62-4218-aa4d-cbc4a78b00fb
md"""
A notebook is a document with two types of cells (see image below): 
- code cells
- markdown cells 

The first type of cells includes computational code, short comments, and Pluto's "begin-end" blocks. Code cells produce an output, which usually takes the form of a plot, a table, numbers, or symbols.

The second type of cell (markdown) allows for the writing of standard text, mathematics, and inserting an image or a video, among other functionalities. 

The versatility that arises from combining markdown cells with code cells makes notebooks a remarkable tool for computational work in general and teaching in particular. The following image provides an overview of most elements composing a notebook. They will be explained in more detail in the next section.
"""

# ╔═╡ dcfb9cc7-6816-4948-a9f9-0481974110da
Resource("https://vivaldomendes.org/images/depot/Pluto_cells.png", :width=>850)

# ╔═╡ 33c4fce7-c7d5-43e4-b9cf-0fac9f2b7fea
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ fb354c4b-3641-4051-ac10-bd8f9559af62
md"""
## 2. Pluto basic tips
"""

# ╔═╡ 555f3004-c66f-4d69-a75a-fc5d47dc14c9
md"""
### Opening a new cell
"""

# ╔═╡ d28dab47-c1d8-4412-8e3f-d0502fe4266a
md"""

- Put the mouse on top of any opened/visible cell. 
- You will see a $(html"<kbd>+</kbd>") sign in the top and bottom left corners (**_see image below_**).
- Click on the $(html"<kbd>+</kbd>") sign where you want the new cell to be placed.
"""

# ╔═╡ 65fb8802-b9d5-4008-80fc-a95f0f53abdd
Resource("https://modernmacro.org/resources/vivaldo/figures/Pluto_basic_tips_Edited.png", :width=>900)

# ╔═╡ d9cd05ac-b1bd-40d6-8a0d-5f50d4788b8d
md"""
### How to run a cell?

- See **_image above_**
- Click on the small circle-arrow in the bottom right-hand-side of each cell $(html"<kbd>▶️</kbd>")
- Or, click simultaneosly on $(html"<kbd>Shift</kbd>") and $(html"<kbd>↩ Enter</kbd>")

"""

# ╔═╡ 7dcfb659-f7d3-40a4-acbe-99aed0299ec7
md"""
### How to save a notebook?

- When a cell is run, this action will be automatically saved
- If we forget to run a particular cell, click simultaneously on $(html"<kbd>Ctrl</kbd>") and $(html"<kbd>S</kbd>") to save all changes that may have been implemented and not yet saved.

"""

# ╔═╡ ad0fa392-3935-459a-9120-81ddcb47baea
md"""
### Code cells
"""

# ╔═╡ ba671d2b-cb42-4ecb-a529-5ca4fe8073be
md"""

- When we click on the $(html"<kbd>+</kbd>") sign, as shown above, a new cell will be automatically opened
- By default, this new cell will be a code cell
- The next cell is an empty code cell that can be used for computation. For example, type $(html"<kbd>2+2</kbd>") and run it.

"""

# ╔═╡ d7a69a24-1c35-403d-9e38-23daf35a013e
2+2

# ╔═╡ b5b67de8-c083-4d12-adc1-ffe3023cc611
md"""
### Markdown cells
"""

# ╔═╡ a2c80a92-5f96-4134-9dd7-60b358f1681f
md"""
- If we want to write text or mathematics in a new cell, or insert a video, we have to turn it into a "markdown" cell
- To change a cell from code mode into **_markdown mode_**, put the cursor inside the cell you want to be in markdown.
  - Click on the following: $(html"<kbd>Ctrl</kbd>") and $(html"<kbd>M</kbd>")  simultaneously. 
- That is all. You can start typing text or mathematics as you like. 

"""

# ╔═╡ d8c7f3dc-3343-4f68-b523-b2f037785634
md"""
!!! tip
	To write text in bold, italics, or mathematical symbols do the following: 
	- type: `**This is bold**` to get: **This is bold**
	- type: `_This is italics_` to get: _This is italics_
	- type: `**_This is bold and italics_**` to get: **_This is bold and italics_**
	- type: `$y=2x^2$` to get a mathematical expression: $y=2x^2$
"""

# ╔═╡ 18727fde-29ed-419e-ad50-0faf1a711484
md"""
#### Bold and Italics
"""

# ╔═╡ f494ad06-2b19-47be-96be-fb6b24cb3aa0
md"""
This is **bold**

This is _italics_

This is **_bold and italics_**

"""

# ╔═╡ 0f9c0bf9-c4c9-492f-8343-551a9cbadfb2
md"""
#### Lists
"""

# ╔═╡ abe31398-0b94-4783-ad8e-fa909fa3d1d3
md"""
- This is my first item

- This is my second item

  - Two tabs give my first sub-item

  - My second

    - Four tabs give first sub-sub-item

    - Very easy

- My final item

1. My first numbered item

2. Whau, this is easy

"""

# ╔═╡ ff171bda-a588-4627-85bd-f0f8f913df93
md"""
#### Writting with mathematical symbols
"""

# ╔═╡ 92aedf0f-c666-4372-85fe-29f28dc6fec2
md"""

To write mathematical expressions in markdown, we only need to use the `$`. Start an expression with a `$` and end it with another `$`.

This is inline mathematics: $y=2x$.

This is displayed mathematics:

$$y=2x$$

This is mathematics using the power syntax: $y = 2x^3$

This is mathematics using the fraction syntax: $h = \dfrac{3}{4n}$

This is more elaborate mathematics: $z = \int_{a}^{b} x^2 dx$

This is ... lack of imagination, using greek symbols:

$$\lambda = \sum_{i=0}^{\infty} \alpha^i x_{i-1}$$
"""

# ╔═╡ 654a151d-9d09-4119-a68d-2e40c533be5a
md"""
#### Tables
"""

# ╔═╡ f1559cf3-db06-43e9-9ba4-35501496f4f0
md"""
It is extremely easy to produce a table. Use the `|` to separate columns and `:` to align columns:

| Model       | Prompt runs | Output parsable | Infrastructure | Result |
|:------      |:-----------:|:---------------:|:--------------:|:------:|
| GPT-4o      | ✓           | ✓               | ✓              | ✅ Works |
| Gemini 2.0  | ✓           | ✓               | ⚠️ Budget      | ⚠️ Partial |
| Gemma 2 27B | ✓           | ✗               | ✓              | ❌ Format |
"""

# ╔═╡ 74010167-4771-4565-99e4-dce565e35e4b
md"""
!!! note "Exercise 1"
 
	- Open a new cell and turn it into markdown mode. Write in this cell the following text: `Today is a sunny day!`
	- Open another cell, turn it into markdown mode, and write the same text but in bold, and also in italics.
	- Open a new cell, turn it into markdown mode, and write the following: This is text and mathematics with a funny equation $y = xz/2$
	- Open a new cell, turn it into markdown mode, and write the following equation: $y = 3x^2$
	
"""

# ╔═╡ 4b438845-97c8-4268-9a58-f97f96f6b93f
md"""
!!! tip "Answer"

	Provide your answer below.	 	 

"""

# ╔═╡ b1f287fe-f2fd-418d-9892-d674105de650
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ aa99585f-4b88-4466-9df6-d0c3d195bbf0
md"""
## 3. Arithmetic operators
"""

# ╔═╡ a1fd4ba1-7c28-4b00-97a3-1ca2913227b0
md"""

	+x	    unary plus	        the identity operation

	-x	    unary minus	      	maps values to their additive inverses

	x + y	binary plus	    	performs addition

	x - y	binary minus		performs subtraction

	x * y	times	        	performs multiplication

	x / y	divide	        	performs division

	x ÷ y	integer divide		x / y, truncated to an integer

	x \ y	inverse divide		equivalent to y / x

	x ^ y	power				raises x to the yth power

	x % y	remainder			equivalent to rem(x,y)'

"""

# ╔═╡ f0b33579-4f78-4537-a70e-073524e22f15
2+3

# ╔═╡ 8ef0f167-66cb-46a6-88d5-7be6bfec46e2
10^4

# ╔═╡ fd300920-4528-413e-807f-c671601b3750
pepe = 10

# ╔═╡ 5808dcf2-8efa-4bc1-93b9-e350f38f38fc
rock = 20

# ╔═╡ 1bdc3f5b-31fc-4abc-a75a-b7c6d0be0165
mary = pepe * rock

# ╔═╡ 6a3abbcf-b0bd-4e7e-a814-8977b112d27c
md"""
`fiona = `$(@bind fiona Slider(-5.0:0.5:5.0, default=1.0, show_value=true))
"""

# ╔═╡ 82294923-9968-4a49-a94f-8ee55fea4664
paty = (fiona * mary , fiona^2 , fiona*mary , 10*fiona)

# ╔═╡ d9d34510-c614-4feb-82a3-ea5e65a6744f
md"""
!!! note "Exercise 2"
 
	What is the value of a variable called "zazu", given the following equation?

	$$zazu = 10 + rock^2 + \frac{pepe}{2}$$
"""

# ╔═╡ e447e90b-cf3e-40c8-8a2b-19dadf7191be


# ╔═╡ a132c1c3-c5d9-461f-b4ac-fd356f16c928
md"""
!!! tip "Answer"

	zazu = ??? 	 	 

"""

# ╔═╡ 37c4d79d-0fbd-49e8-9166-af5f557acabf
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 487fe09a-2079-407d-8cee-6e2db4fe4898
md"""
## 4. Ploting functions
"""

# ╔═╡ c602906d-1116-497e-a708-2031e04014e2
begin
	x =  0.0 : 0.01 : 20.0
	#Ana = 2 * x.^0.5
	Ana = 2 * sin.(x)
end;

# ╔═╡ 6cbcbbe1-e50c-4590-8261-dc03e4240a8f
begin
	fig1 = plot(x, Ana)
	#relayout!(fig1, Layout(height = 450, hovermode="x", title_text="My fancy plot", title_x = 0.5, titlefont_size="17"))
	##fig1
end

# ╔═╡ 4824f2fe-7c60-43d8-9211-87bfa7aa4065
noise = randn(500)

# ╔═╡ b0373101-a51a-4b52-8ba5-662530a744c9
begin
	fig3 = plot(histogram(x=noise, opacity=0.75,  nbinsx=50), Layout(bargap= 0.05))
end

# ╔═╡ da2bd2d5-c1a5-460b-8240-b3135aa123b1
md"""
!!! note "Exercise 3"
 
	In the following cell, produce a simple plot of the following function: $$e^{Ana}$$ 
	
	by using the following code:`Plot(x, exp.(Ana))`
"""

# ╔═╡ 50de0c4e-6308-488e-b520-805dc9231349
plot(x, exp.(Ana))

# ╔═╡ 4970e6ab-5b01-418f-b22f-b9641ac4d53f
md"""
!!! note "Exercise 3A"
 
	In the next cell, we hide the code that plots three functions together. To run the cell, click on the little eye to bring the cell back to the surface, delete the `;` found in the last line, and finally run the cell (`Shift` and `Enter`, or click on the `run cell` button). The plot will then be produced.
"""

# ╔═╡ a5b1d8ec-23c8-4597-84c0-0b7c93cb5d30
begin
	x2 = LinRange(0, 3*π, 100)
	
	trace1 = scatter(x = x2, y = cos.(x2), line_width = 2, line_color = "blue", name = "Shrek")
	
	trace2 = scatter(x = x2, y = sin.(x2), line_dash = "dashdot", line_color="BlueViolet", name="Fiona")
	
	trace3 = scatter(x = x2, y = cos.(x2).^2 .- 1/2, mode = "markers+lines", marker_symbol = "circle-open",
	 	 			marker_size = "6", marker_color = "red", name = "Donkey", line_width = 0.5)
	
	layout1 = Layout(#font_size = 13, 
				height = 450,
				hovermode="x",	
			   	title_text="My fancy plot with different lines & marker styles", title_x =0.5,
			   	titlefont_size="17")
	
	fig4 = plot([trace1, trace2, trace3],layout1)
end

# ╔═╡ ff8cd19e-8cae-47b3-8c8e-2f127779e570
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 9249b41d-3e4a-4116-b9be-65636e68cac7
md"""
## 5. Magic things we can do with Pluto
"""

# ╔═╡ 8dbe4a85-3bb0-442f-8618-720c0a00af67
md"""
### Write normal text and sophisticated mathematics
"""

# ╔═╡ e6983682-02a8-4d86-b76e-c85235931ebe
md"""
Consider now an optimization problem where the constraints are defined in terms of inequalities. This can be solved by using the famous Karush-Kuhn–Tucker conditions, which require too much space to be explained here. Ax encellent place to look at is Chapter 6 of the textbook: Rangarajan K. Sundaram  (1996). _A First Course in Optimization Theory_, Cambridge University Press.

Using two packages of Julia (JuMP and GLPK), the syntax to solve this problem can be found below. In particular, look how close the Julia syntax is to the mathematical syntax.
"""

# ╔═╡ 0de12035-fc43-445e-a6f8-f97d7234de99
md"""
$$\max \quad x_{1}+2 x_{2}+5 x_{3}
$$

subject to

$$
\begin{aligned}
-x_{1}+x_{2}+3 x_{3} & \leq-5 \\
x_{1}+3 x_{2}-7 x_{3} & \leq 10 \\
0 \leq x_{1} & \leq 10 \\
x_{2} & \geq 0 \\
x_{3} & \geq 0
\end{aligned}$$

And then, blaa blaa blaa ....
"""

# ╔═╡ fee19510-7829-4a1a-b95e-b67b0cdd6798
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ ae7d434b-4623-497e-9fe4-c3b9f3f10e69
md"""
### Use fancy symbols for writing text and doing computation
"""

# ╔═╡ 7212caf4-6d99-4c01-92f4-c3f11be7d446
md"""
I have a 😺 that loves the 🌚, but not 🍍. And I can do write simple text or do computations with these adorable objects. 
"""

# ╔═╡ 95b13f99-c77f-41a2-ab26-8c405296fd17
md"""
!!! tip
	To write special characters (like beta, for example) do the following: 
	- type: `\beta` followed by `Tab`
	- The firt time you do it, you have to click more than one time on the `Tab` key. After that, clicking only once will be enough.
"""

# ╔═╡ cb8fbf86-4646-4888-993d-7f602af2183f
β = 3

# ╔═╡ 0b13ae26-c9ea-440d-b23c-af32664c4730
α = 10 * β

# ╔═╡ cfb5f1f7-9e82-4b64-bede-4c93dc166c20
β^2

# ╔═╡ 0cb64be3-d306-4b2d-8ebf-020aaeb36a11
🍎=5

# ╔═╡ ef67c2d4-50e5-4a75-9d1d-7f20d05ff38f
2*🍎

# ╔═╡ d456b292-6c76-4484-bf76-f74490beb720
😃 = 500

# ╔═╡ e8b65fdc-7a03-4cb8-9865-2e168989b2a8
🍎*🍎 / (2*😃)

# ╔═╡ 37f16fc1-5450-4087-a4f4-72a03db847d4
md"""
!!! note "Exercise 4"
 
	- Open a new cell and turn it into markdown mode. 
	- Write in this cell the following symbols (beta, psi, theta): β ψ θ
"""

# ╔═╡ 41d9598c-3011-4e14-8684-289a172cec10


# ╔═╡ bd1b97c5-04ee-499b-81d7-4bf11e1a4d07
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ c23a2871-6108-4c15-853a-9f2deb41ad53
md"""
### Play with sliders
"""

# ╔═╡ c7c5947b-4caf-4278-af25-c6efdf37557a
md"""
!!! note "Exercise 5"
 
	Move the two sliders above (one slide at a time), and see what happens to the equilibrium in the plot above.
"""

# ╔═╡ 4ba352ad-7f94-4f00-ae90-49c92226d0ae
md"""
	!!! tip "Answer"

		The equilibrium will ....  	 	 

	"""

# ╔═╡ 9c6874cf-6dee-489f-b9ed-89e86ecdb865
md"""
!!! note "Exercise 6"
 
	Move the two sliders above (one slide at a time), and see what happens to the sinusoidal wave below.
"""

# ╔═╡ e1bcc8dd-009f-4f82-832f-0d8db11b700d
md"""
#### Calculate an integral
"""

# ╔═╡ 2557c704-295a-4770-bbeb-41a94257738b
md"""
!!! note "Exercise 7"
 
	We can estimate the integral of a function $F:[a,b] \rightarrow \mathbb{R}$ using the midpoint rule approximation. Let us interactively estimate the $\int_0^1 F(x) d x$ to see the effect of changing the number of subintervals, in the case of:

	$$F(x)= - xsin(2\pi x)$$
"""

# ╔═╡ 4000f0d1-f014-4ed2-bb6e-de637e47a76d
begin
	Nbins_slider = @bind Nbins html"<input type=range min=1 max=500 step=1 value=20>"
		
	md"""Number of intervals: $Nbins_slider"""
end

# ╔═╡ 35c1d390-6b67-4ae4-815b-fd5eebdb1ee2
let
	F(x)=-x*sin(2π*x) 
	centres=1/(2*Nbins):1/Nbins:1
	midpointarea=round(sum(F.(centres))/Nbins, digits=6)
	
	trace01 = bar(x=centres, y=F.(centres), kind="bar", name = "Area")
	trace02 = scatter(x=centres, y= .- centres .* sin.(2 .* π .* centres), name = "F(x)")
		
	plot([trace01, trace02], Layout(height = 450, title_text="midpoint rule estimate with $Nbins intervals = $midpointarea"))
end

# ╔═╡ 04c46c48-a297-4955-90bd-a51b6ad9b310
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ a69868ac-f4e5-4125-9584-7a0750180161
md"""
### Playing with questions & answers 
"""

# ╔═╡ c16f9c18-e776-45c8-bcff-66c78627928d
md"""
!!! note "Exercise 8"
 
	Table 1 shows some important macroeconomic aggregates, measured in **real terms**, for the US economy in 2013. The data in this table has been inserted into the notebook (see cell below). Answer the questions 👉 **a)** and 👉 **b)**
"""

# ╔═╡ 1895d767-4018-43a8-94cd-174f321daca1
md"""
$$\begin{aligned}
& \text {Table 1. Expenditure in the USA in 2013}\\
&\begin{array}{cllr}
\hline \hline \text {  } & \text{Code} & \text {Headings} & \text {Billions of dollars} \\
\hline  
1 & cons & \text{Private Consumption Expenditures} & 11400 &  \\
2 & inve & \text{Investment in Equipment and Structures} & 2600 &  \\
3 & stoc & \text{Inventories} & 80 &  \\
4 & gove & \text{Public Expenditures} & 3120 &  \\
5 & expo & \text{Exports} & 2250 &  \\
6 & impo & \text{Imports} & 3000 &  \\

\hline
\end{array}
\end{aligned}$$

"""

# ╔═╡ 05ce80f3-67e6-4e61-9447-4d703e1c67eb
# values passed into the notebook
cons = 11400 ; inve = 2600 ; stoc = 80 ; gove = 3120 ; expo = 2250 ; impo = 3000 ;

# ╔═╡ edb2c820-0b38-4f4c-8779-b0f45815b086
md"""
👉 **a)** Using Table 1, calculate GDP by the expenditure method. 
"""

# ╔═╡ ad797a54-db6f-4d85-aed2-f8f8ec774c0e
md"""
!!! hint

	According to the usual definition of GDP by the expenditure method, GDP is equal to the sum of  private consumption, plus public expenditure on goods and services, plus investment and inventories, plus exports minus imports.
"""

# ╔═╡ 12692d0c-2a1f-4ef2-b529-451faa7c9b00
GDP = cons + inve + stoc + gove + expo - impo

# ╔═╡ 057e1b24-e9fa-4aa5-b9b1-97181b096851
md"""
	!!! tip "Answer (a) "

		GDP = ??? 

	"""

# ╔═╡ 2c93ad8b-c489-4d37-a39f-a116715f1cfc
md"""
👉 **b)** In a market economy, such as the one we live in, do you have any suggestion about the main macroeconomic variables that explain the behavior of private investment?
"""

# ╔═╡ 5c83f904-5ff8-4056-b6a7-dc4ea8b3c993
md"""
!!! tip "Answer (b) "

	We have not yet discussed the main determinants of private investment in our economies. As we will see later in this course, those determinants are: real interest rates, financial frictions, and autonomous investment decisions.

"""

# ╔═╡ c644bcd3-afef-4446-ae11-2cce4b37eb88
md"""
!!! note "Exercise 8A"
 
	We can use Pluto to create multiple-choice questions with some pedagogic feedback. Try to answer the following question using the bullets below.

"""

# ╔═╡ cb273532-f78e-48f6-9a32-5bbfc05164dc
md"""
How much did the US GDP change in 2023, given the following facts:

- First Republic Bank shareholders lost 2 billion dollars on March 16 due to the bank's collapse.

- Microsoft decided to invest 10 billion dollars in the development of ChatGPT (OpenAI). 

- Amazon bought 100% of the capital of the company One Medical on February 22, 2023 for 4 billion dollars.


**a)** -- 2 billion + 10 billion + 4 billion.

**b)** + 4 billion.

**c)** + 10 billion + 4 billion.

**d)** + 10 billion.

**e)** None of the above.
"""

# ╔═╡ 7f592271-13b2-4d4e-abe7-495172544e14
@bind q20 Radio(["a)","b)","c)","d)","e)"])

# ╔═╡ fe2181e1-ef9f-4b6a-aff2-a06f41f166df
md"""
!!! note "Exercise 8B: demand vs supply in a market economy"
 
	As you have studied in introductory microeconomics, price adjustments are the mechanism that equilibrates markets. The following figure illustrates the interaction between supply and demand in a market economy that leads to such adjustments.

	What kind of information can we get by looking at points 1 and 1'? And what happens in the case of points 2 and 2'?

"""

# ╔═╡ 35edcb27-2cde-4fc8-a2ba-7f455493c627
begin
	Q = 0.0 : 0.05 : 25.0
	Pˢ = 0.0 .+ 0.75 .* Q
	Pᵈ = 18.0 .- 0.75 .* Q

	trace9a = scatter(; x = Q, y = Pᵈ,  mode="lines" , line_color = "maroon", line_width = "2.2",	name  = "Qᵈ")
	
	trace9b = scatter(; x = Q, y = Pˢ, mode="lines" , line_color = "Blue", line_width = "2.2",	name  = "Qˢ")


	trace4_2 = scatter(; x =[11.0 , 13.0],  y = [8.25 , 8.25], text =["1", "1'"], textposition = "top center", 
                    name ="", mode="markers+text", marker_size= "11", marker_color="purple", showlegend = false, 		 					hoverinfo="skip", textfont = attr(family="sans serif", size=16, color="black"))

	trace4_3 = scatter(; x =[11.0 , 13.0],  y = [9.75 , 9.75], text =["2", "2'"], textposition = "top center", 
                    name ="", mode="markers+text", marker_size= "11", marker_color="darkorange", showlegend = false, 		 					hoverinfo="skip", textfont = attr(family="sans serif", size=16, color="black"))
	
		
	layout9 = Layout(;
					height = 450,
					title_text ="Demand (Qᵈ) versus supply (Qˢ)",
					title_x = 0.5,
					hovermode="y unified",
					#hovermode = "y",
                    xaxis=attr(title="Quantity", showgrid=true, zeroline=false),
                    xaxis_range = [9.0, 15.0],	
                    yaxis=attr(title="Price", zeroline = false),
					yaxis_range=[6.5 , 11.5]
	)

   fig9 = plot([trace9a, trace9b, trace4_2 , trace4_3], layout9)	
end

# ╔═╡ 2b922c32-d0da-4813-b1ad-abb5167f4ce4
md"""
!!! tip "Answer 8B"

	Points 1 and 1' tell us that for the price level of 8.25, there is  .... .

	Points 2 and 2' tell us that for the price level of 9.75, there is a .... .

	The price will be stable when there is .... .

"""

# ╔═╡ 1fa4dabd-3459-4187-b9f2-7e1eee56d28a
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 8391202d-cec8-4b0c-bfc2-1a01419851d4
md"""
## 6. Downloading and plotting data
"""

# ╔═╡ a1cffba2-aab6-4ce5-b5c1-fe7f55f397f9
md"""
###### Students are not expected to know *_how to do_* the plots below, but they must know *_how to use them_* if asked.
"""

# ╔═╡ f669ec62-dca0-42c9-a7b7-8024f6e9c7a9
md"""
To download the data file attached to this notebook -- its name is _`ECB Data Inflation.csv`_ -- we should do as exemplified below. A CSV file is the most used format to deal with data and it means "comma separated values". This data file contains the inflation rate for the EuroZone as a whole and four individual countries. This data was obtained from [here](https://ec.europa.eu/eurostat/databrowser/view/prc_hicp_manr/default/table?lang=en).

```julia
name_my_data = CSV.read("ECB Data Inflation.csv", DataFrame) 	
```
"""

# ╔═╡ 390010ce-72dd-4b69-a486-7bcc1744859f
inflation = CSV.read("ECB Data Inflation.csv", DataFrame)

# ╔═╡ 43a443c2-3f6d-426f-ab72-bbd67873b92e
md"""
To summarize the data we downloaded, we should do:

```julia
describe(name_my_data)
```
"""

# ╔═╡ 49f95b2d-6186-470b-956c-1e6bb3ae3de2
describe(inflation)

# ╔═╡ abf832f5-7cb4-40e7-a27e-2bdec69c910a
md"""
To plot a single column of `inflation` we can do:
"""

# ╔═╡ cd6c0607-ad69-455d-bc3b-6ce05fc87746
begin
	fig5 = plot(inflation.Months, inflation.Portugal, Layout(height = 450))
end

# ╔═╡ 78e2f78e-061e-4a92-8cce-4c82a46c6e31
md"""
To plot more than one column, we can create a vector `[...]` with as many columns has we wish. In the case below we plot the columns associated with Germany and Portugal.
"""

# ╔═╡ 1a780e63-d0aa-4ad5-a76f-0314c50e3c29
begin
	fig6 = plot(inflation.Months, [inflation.Germany  inflation.Portugal], Layout(height = 450))
end

# ╔═╡ be81679d-e8b1-4dc4-b7d1-489fe363f37f
md"""
We can add a title, the names of the variables (columns), a label to both axis, and a hover functionality into our previous plot. It will look like this (the code is hidden: to show the code click on the little eye button on the left):
"""

# ╔═╡ 49a6d562-d47c-470b-ae54-601fffd12c68
begin
	restyle!(fig6, 1:2, name = ["Germany", "Portugal"]) # names Germany and Portugal on variables 1 and 2
	
	relayout!(fig6, Layout(
		title_text = "Inflation in Germany and Portugal", title_x = 0.5, 
		hovermode = "x", yaxis_title = "Percentage points", 
		xaxis_title = "Monthly observations")) # introduces a title, y-axis label, and the hover-mode
	
	fig6 # Updates the plot
end

# ╔═╡ be1e91a5-011d-425a-be16-ab7b45b6823f
md"""
!!! note "Exercise 9"
 
	We can also save one plot as a "png" file or an "svg" file (among many other formats). These files can be inserted later into another document (like a PowerPoint or a Word file). Their graphical quality will be much higher than by using the archaic "screen-capture" functionality. In the next cell, save "fig6" as a svg file by copying and running the code `savefig(fig6, "inflation.svg")`. It will be automatically saved in the same folder where this notebook is.

"""

# ╔═╡ 9362414b-e883-4569-9f31-f4ce08d9a0e5
savefig(fig6, "inflation.svg")

# ╔═╡ 9c4a07a8-4144-4e01-ae25-d35c60db5951
begin
	trace1_1 = scatter(;x = inflation.Months, y = inflation.EuroArea, 
				name = "EuroArea", mode="markers+lines",
				marker_symbol="circle", marker_size="4",line_width= 0.3, 
            	marker_color = "Blue")
	
	trace1_2 = scatter(;x = inflation.Months, y = inflation.Portugal, 
				name = "Portugal", mode="markers+lines",
				marker_symbol="circle", marker_size="4",line_width= 0.3,
            	marker_color = "Red")

	trace1_3 = scatter(;x = inflation.Months, y = inflation.Germany, 
				name = "Germany", mode="markers+lines",
				marker_symbol="circle", marker_size="4",line_width= 0.3,
            	marker_color = "Gray")

	trace1_4 = scatter(;x = inflation.Months, y = inflation.Spain, 
				name = "Spain", mode="markers+lines",
				marker_symbol="circle", marker_size="4",line_width= 0.3,
            	marker_color = "Purple")

	trace1_5 = scatter(;x = inflation.Months, y = inflation.France, 
				name = "France", mode="markers+lines",
				marker_symbol="circle", marker_size="4",line_width= 0.3,
            	marker_color = "Orange")

	layout1_2 = Layout(;#font_size = 16,
			height = 450,
					
			title_text = "Inflation in the EuroZone and Portugal", title_x=0.5,
		
			hovermode="x",		
		
            xaxis = attr(
               title = "Monthly obervations",
               tickformat = "%Y",
               hoverformat = "%Y-M%m",
               tick0 = "1997/01/01",
               dtick = "M60" ),
		
        	xaxis_range = [Date.(1997), Date.(2024)],
        	yaxis_title = "Percentage points",
        	#yaxis_range=[-2, 2], 
        	titlefont_size = 16)

	fig7 = Plot([trace1_1, trace1_2, trace1_3, trace1_4, trace1_5], layout1_2)
end

# ╔═╡ f84e13af-a1c3-40a5-80a6-b838410b24c7
md"""
!!! note "Exercise 9A"
 
	Create a simple plot of the following variable `inflation.EuroArea .- 2` (EuroArea inflation minus 2% inflation). To achieve that, apply the following code `Plot(inflation.Months, inflation.EuroArea .- 2, kind="bar")` , where `kind="bar"` means that we want the data to be plotted as a bar, not as a line. The level of inflation the European Central Bank wants to see in the Euro Zone is 2%. Check whether the wishes of the ECB have been accomplished in the EU.

"""

# ╔═╡ 11a363b2-90c7-43ac-a563-257056c66908
plot(inflation.Months, inflation.EuroArea .- 2 , kind = "bar", Layout(height = 450))

# ╔═╡ f1fd0927-4d06-41df-8de4-6def18e26a22
md"""
!!! tip "Answer"

	Between 2000 and mid-2007, the ECB was successful in maintining inflation very close to the desired level: 2%. Since then, the ECB has shown tremendous difficulty maintaining inflation close to its desired level.		 

"""

# ╔═╡ da1d8381-5b15-475e-a693-4696689fa6d3
md"""
## 7. Using linear algebra 
"""

# ╔═╡ fdb13094-ef0e-4626-9e57-4110c0d2ccae
md"""
!!! warning

	In computation, there is a crucial difference between a `,` and a `.`. Therefore, do not forget that $1/2=0.5$, NOT $1/2=0,5$. If you mistakenly use a `,` to express decimal units, you will create a problem your computer does not know how to solve.

"""

# ╔═╡ bcc72069-c9c9-4eb9-96a1-8528cd60c100
A = [1 2 3 ; 4 1 6 ; 7 8 1]

# ╔═╡ fa1e19f1-9cbd-4454-9bf1-a1fc8cdcf52a
inv(A) # the inverse of A

# ╔═╡ 5efc650a-92a5-4350-9eda-3e7b61f491f5
transpose(A) # the transpose of A

# ╔═╡ ad3bb2f0-b8b7-489d-8603-53ebd01ab86f
det(A) # the determinant of Bia

# ╔═╡ 2bb008a1-82e2-487f-b2f5-0d27eea804de
A'

# ╔═╡ 789cab47-ef60-4c0c-b380-42d9fadee744
A^2

# ╔═╡ b2bc41af-aba3-484f-97ed-5558987f4d93
Zak = randn(3, 3)

# ╔═╡ 531d6c8a-c325-4e9c-9496-6c69b0abed6b
aple = A * Zak

# ╔═╡ c8bdc59a-b830-41d7-903e-338a17220d18
F=rand(1:10, 5,5)

# ╔═╡ 3a6ae0de-3774-43d1-aad0-0c1e3edd99e8
F[1,3]

# ╔═╡ 0a09e61c-7c2b-4088-9fb8-465189ba1768
md"""
!!! note "Exercise 10"
 
	Let us create a matrix of dimension (100x100), with all its elements as randomly generated numbers. Give this matrix a funny name Shrek: `Shrek = randn(10,10)`

	Calculate the inverse of Shrek. How long does it take your computer to calculate such an inverse matrix?
"""

# ╔═╡ c9b02af3-34ca-4f74-8611-0112b83942bb
Shrek = randn(10,10)

# ╔═╡ 2588a536-5bfd-4af8-8b40-1a918792cd6a
inv(Shrek)

# ╔═╡ b0a9c155-65f7-44f3-a9bf-77447da741ab
md"""
!!! tip "Answer"

	The time it takes to compute the inverse of Shrek (a very large matrix) is around 250 microseconds (220 µs). It is almost instantaneous.

"""

# ╔═╡ 34feb3a2-01a2-48a3-87c7-47000b18c39c
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 93ba50b7-098e-4b3d-a3d4-ad27a5d67e0f
md"""
## 8. Solving a complicated model
"""

# ╔═╡ 5766096f-96dc-44f9-ae72-4f75f441594f
md"""
!!! note "Exercise 11"
 
	Consider the following prototype of a macroeconomic model:

	$\begin{aligned}
	& y_{1} = a + 0.5 y_{2} - 2.5 y_{3} \\
	& y_{2} = -2 y_{1} + 5 y_{3} + b \\
	& y_{3} = 10 y_{1} + 2 c \\
	& a = 10 \ , \ b=2 \ , \ c=5
	\end{aligned}$

	Let us solve this model using the package `NLsolve`. The first thing we should do is to pass the values of the variables with known values to the notebook: $a=10,b=2,c=5$: 

"""

# ╔═╡ 4e40b2ae-3e70-47bc-808f-1acd2615615a
a = 10 ; b = 2 ; c = 5 ; # the data is passed into the notebook

# ╔═╡ 458274f4-636a-4d5b-893d-33f262aed1fa
md"""
The first thing we have to is to write our model as an homegeneous system (zeros on its left hand side)

$$\begin{aligned}
&0=a+0.5 y_2-2.5 y_3 -y_1 \\
&0=-2 y_1+5 y_3+b -y_2 \\
&0=10 y_1+2 c - y_3
\end{aligned}$$
"""

# ╔═╡ 7b15c7ca-1e90-4403-8bcb-ba0194ff1950
md"""
Then, we have to have to write down our problem according to the syntax of NLsolve: 

					Name, Variables, Functions, Solution

- **Name**. Give a name for our problem. Let us name it as _shrek _ problem_

- **Variables**. Define the variables in our problem: $(html"<kbd>v[1]</kbd>"), $(html"<kbd>v[2]</kbd>"), $(html"<kbd>v[3]</kbd>")

- **Functions**. Define the functions in our problem: $(html"<kbd>F[1]</kbd>"), $(html"<kbd>F[2]</kbd>"), $(html"<kbd>F[3]</kbd>")

- **Solution**. Compute the solution according to the syntax of $(html"<kbd>NLsolve</kbd>")

  - Give a name to the solution to our problem: let us name it as $(html"<kbd>Shrek_solution</kbd>")

  - Call the NLsolve function that will solve the problem (always this one): $(html"<kbd>nlsolve</kbd>")

  - Choose the problem to be solved (in this case, is $(html"<kbd>shrek_problem</kbd>") )

  - Give three initial guesses for the package to start the computation. 

      - For linear problems $(html"<kbd>[1.0  ; 1.0  ; 1.0]</kbd>") always works. 

      - The number of initial guesses must equal the number of unknowns in the model.

In the image below, we provide a detailed step-by-step explanation of how the solution to the problem above is implemented.
"""

# ╔═╡ bdd4566f-3800-4c9c-b91e-ac3c5a65580a
HTML("""
<div style="display: flex; justify-content: center; align-items: center;">
    <img 
		src="http://modernmacro.org/resources/Vivaldo/figures/NLsolve_explained.png" 
	alt="Centered Image" style="max-width: 90%; height: auto;">
</div>
""")

# ╔═╡ 4a4cf9f2-21b5-4685-9ce8-99f76556811c
begin
	#defining the problem
	function shrek_problem(F, v)

		# defining variables
		y1 = v[1] 
        y2 = v[2] 
		y3 = v[3] 

		# deining functions
        F[1] = a + 0.5y2 - 2.5y3 - y1
        F[2] = -2y1 + 5y3 + b - y2
		F[3] = 10y1 + 2c - y3 		
	end

	# computing the solution
	shrek_solution = nlsolve(shrek_problem, [0.0 ; 0.0 ; 0.0])		# computes the solution to the problem 
	shrek_solution.zero  											# rounds off the solution
end

# ╔═╡ 09689df6-67a3-4f6e-aeb3-7bacc4d738e3
md"""
!!! tip "Answer"

	The solution is: y1 = 5.5, y2 = 316, y3 = 65.

"""

# ╔═╡ 79b30313-e600-4356-b589-69fbb9bdc04d
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 7d10fdd7-197f-45f8-b316-c4774b0a1f79
md"""
## Auxiliary cells (please do not change them)
"""

# ╔═╡ c306ae6c-8acc-440d-94f2-4d17c01fa999
TableOfContents()

# ╔═╡ 9c386789-44b7-4d38-95ca-beeaecb311b0
begin	
	# Demand	
	Ā3 = 8.0
	λ3 = 0.5
	m3 = 2.0
	ϕ3 = 0.2
	r̄3 = 2.0	
	# Supply	
	γ3  = 2.8
	Yᴾ3 = 14.8
	πᵉ3 = 2.0
	ρ3  = 0.0	
	# Intercepts (useful to simplify notation)	
	πd_max3 = (Ā3 / (λ3 * ϕ3)) - r̄3/λ3        # The AD curve y-axis' intercept
	πs_min3 = πᵉ3 - γ3 * Yᴾ3 + ρ3             # The AS curve y-axis' intercept
end;

# ╔═╡ 13054251-e17d-4469-8a65-281ffd3b5999
begin
	π_ZL3= -r̄3 / (1+λ3)
	πd_min3 = - Ā3 / ϕ3
	Y_ZL3 = Ā3 * m3 - (m3 *  ϕ3 * r̄3) / (1+λ3)
	ρ4 = -3.15
end;

# ╔═╡ df7041ea-07b6-4e13-b463-b6fa7b95111c
function tchanã(ans::Union{Nothing,String},correct::String)	
	if typeof(ans) == Nothing
		md"""
		👆 *Please, select the correct answer.*
		
		---
		"""
	else
		if ans == correct
			md"""
			!!! tip "🎉 Corret!"

			---
			"""
		else	
			md"""
			!!! danger "👎 Wrong answer"
				
				Try again. **But it is better to study first!!!** 😡

			---
			"""
		end
	end
end;

# ╔═╡ 8419b89c-80c1-427e-bf68-03c3a07e8ef2
tchanã(q20,"d)")

# ╔═╡ cac4054f-21df-4aac-be8e-ac11e30a1181
md"### CSS code"

# ╔═╡ 47831542-8188-496d-b712-b8fc4db18694
begin
      struct TwoColumns{L, R}
    left::L
    right::R
      end
      
      function Base.show(io, mime::MIME"text/html", tc::TwoColumns)
          write(io, """<div style="display: flex;"><div style="flex: 20%;margin-left:19%;">""")
          show(io, mime, tc.left)
          write(io, """</div><div style="flex: 20%;">""")
          show(io, mime, tc.right)
          write(io, """</div></div>""")
      end
end

# ╔═╡ c1d9b788-0454-4fb4-9eca-f08a8f693af5
TwoColumns(
	
md"""

`ΔĀ = `$(@bind ΔĀ Slider(-1.0:0.01:1.0, default=0.0, show_value=true))

""",
	
md"""

`ΔYᴾ = `$(@bind ΔYᴾ Slider(-1.0:0.05:1.0, default=0.0, show_value=true))

"""
)

# ╔═╡ afbb7662-85e1-4a35-b82f-bce445cf0936
begin
	Y3 = 13.8:0.01:15.6
	
	πd3 = πd_max3 .- ((1 ./(m3 .* ϕ3 .* λ3)) .* Y3) # AD function
	πs3 = πs_min3 .+ γ3 .*Y3             # AS function
	
	trace9_0 = scatter(; x = [14.8 .+ ΔYᴾ , 14.8 .+ ΔYᴾ] , y = [-5,10] , 
			mode = "line", line_width = "4", name = "Yp", line_color = "Orange")
	
	trace9_1 = scatter(; x = Y3, y = πd3, mode="lines" , line_color = "Blue", line_width = "3",
			name  = "AD")
	
	trace9_2 = scatter(; x = Y3, y = πs3, mode="lines" , line_color = "Red", line_width = "3",
					name  = "AS")
	
	trace9_3 = scatter(; x = [14.8], y = [2.0], text =["1"], 
					textposition = "top center", name ="Eq1", mode="markers+text", marker_size= "12",
					marker_color="Blue", textcolor = "Black",
	 				textfont = attr(family="sans serif", size=18, color="black"))


	πd_max3_ΔĀ = ((Ā3 .+ ΔĀ) ./ (λ3 .* ϕ3)) .- r̄3 ./ λ3 # New max value of the AD function
	
	πd3_ΔĀ = πd_max3_ΔĀ .- ((1 ./(m3 .* ϕ3 .* λ3)) .* Y3) # AD function
	
	
	trace9_4 = scatter(; x = Y3, y = πd3_ΔĀ, mode="lines" , line_color = "Magenta", 
					 line_width = "3", name  = "AD2")
		
	#trace9_5 = scatter(; x = [14.21], y = [2.947], text =["2"], 
	#				textposition = "top center", name ="Eq2", mode="markers+text", 
	#				marker_size= "12", marker_color="Magenta",
	#				textfont = attr(family="sans serif", size=16, color="black"))
	
	
	layout9_5 = Layout(;
					height = 450,
					title_text ="Initial long-term equilibrium",
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" GDP trillion dollars (Y)", showgrid=true, zeroline=false),
                    xaxis_range = [14.2, 15.4],	
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-0.8 , 5])

   fig0 = plot([trace9_0, trace9_1, trace9_2, trace9_4, trace9_3], layout9_5)
	
	
end

# ╔═╡ cca6bd64-0735-4bd4-8149-7f03f7f26e57
TwoColumns(
	
md"""

`frequency: `$(@bind frequency Slider(0.0:0.1:3.0, default=1.0, show_value=true))

""",
	
md"""

`damping:  `$(@bind damping Slider(-0.5:0.01:0.8, default=0.0, show_value=true))

"""
)

# ╔═╡ cffd033c-9ba9-4147-922e-2265b9e46485
begin
	τ=1:0.01:20
	plot(τ, (exp.(.-damping * τ) .* sin.(frequency .* τ)), Layout(height = 450))
end

# ╔═╡ d65ae8f6-07be-48f7-a1c8-fc493c5cde43
html"""
<style>
	@media screen {
		main {
			margin: 0 auto;
			max-width: 1550px;
    		padding-left: max(200px, 10%);
    		padding-right: max(383px, 10%); 
            # 383px to accomodate TableOfContents(aside=true)
		}
	}
	form {
		text-align: center;
		width: fit-content;
		padding: 10px;
  		border: 5px solid #ebebeb;
  		margin: 1em 0 2.5em 0;
		border-radius: 25px;
  		background: #ebebeb;
		color:#1f1f1f;
	}
	form > div {
		display: inline-block;
		margin-left: 0.75em;
		margin-right: 0.75em;
		font-weight: bold;
	}
</style>
"""


# ╔═╡ 00000000-0000-0000-0000-000000000001
PLUTO_PROJECT_TOML_CONTENTS = """
[deps]
CSV = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
ColorSchemes = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
CommonMark = "a80b9123-70ca-4bc0-993e-6e3bcb318db6"
DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
Dates = "ade2ca70-3891-5945-98fb-dc099432e06a"
HypertextLiteral = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
LinearAlgebra = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
NLsolve = "2774e3e8-f4cf-5e23-947b-6d7e65073b56"
PeriodicalDates = "276e7ca9-e0d7-440b-97bc-a6ae82f545b1"
PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
PlutoPlotly = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
PlutoTeachingTools = "661c6b06-c737-4d37-b85c-46df65de6f69"
PlutoUI = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
ShiftedArrays = "1277b4bf-5013-50f5-be3d-901d8477a67a"
SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
Statistics = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
StatsBase = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"

[compat]
CSV = "~0.10.15"
ColorSchemes = "~3.31.0"
CommonMark = "~0.10.0"
DataFrames = "~1.8.1"
HypertextLiteral = "~0.9.5"
NLsolve = "~4.5.1"
PeriodicalDates = "~2.0.0"
PlotlyKaleido = "~2.3.1"
PlutoPlotly = "~0.6.5"
PlutoTeachingTools = "~0.4.7"
PlutoUI = "~0.7.79"
ShiftedArrays = "~2.0.0"
StatsBase = "~0.34.10"
"""

# ╔═╡ 00000000-0000-0000-0000-000000000002
PLUTO_MANIFEST_TOML_CONTENTS = """
# This file is machine-generated - editing it directly is not advised

julia_version = "1.10.10"
manifest_format = "2.0"
project_hash = "c7492678fed3096232917ad0d4b9adc015c63baa"

[[deps.ADTypes]]
git-tree-sha1 = "f7304359109c768cf32dc5fa2d371565bb63b68a"
uuid = "47edcb42-4c32-4615-8424-f2b9edc5f35b"
version = "1.21.0"

    [deps.ADTypes.extensions]
    ADTypesChainRulesCoreExt = "ChainRulesCore"
    ADTypesConstructionBaseExt = "ConstructionBase"
    ADTypesEnzymeCoreExt = "EnzymeCore"

    [deps.ADTypes.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ConstructionBase = "187b0558-2788-49d3-abe0-74a17ed4e7c9"
    EnzymeCore = "f151be2c-9106-41f4-ab19-57ee4f262869"

[[deps.AbstractPlutoDingetjes]]
deps = ["Pkg"]
git-tree-sha1 = "6e1d2a35f2f90a4bc7c2ed98079b2ba09c35b83a"
uuid = "6e696c72-6542-2067-7265-42206c756150"
version = "1.3.2"

[[deps.Adapt]]
deps = ["LinearAlgebra", "Requires"]
git-tree-sha1 = "7e35fca2bdfba44d797c53dfe63a51fabf39bfc0"
uuid = "79e6a3ab-5dfb-504d-930d-738a2a938a0e"
version = "4.4.0"

    [deps.Adapt.extensions]
    AdaptSparseArraysExt = "SparseArrays"
    AdaptStaticArraysExt = "StaticArrays"

    [deps.Adapt.weakdeps]
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.AliasTables]]
deps = ["PtrArrays", "Random"]
git-tree-sha1 = "9876e1e164b144ca45e9e3198d0b689cadfed9ff"
uuid = "66dad0bd-aa9a-41b7-9441-69ab47430ed8"
version = "1.1.3"

[[deps.ArgTools]]
uuid = "0dad84c5-d112-42e6-8d28-ef12dabb789f"
version = "1.1.1"

[[deps.ArrayInterface]]
deps = ["Adapt", "LinearAlgebra"]
git-tree-sha1 = "d81ae5489e13bc03567d4fbbb06c546a5e53c857"
uuid = "4fba245c-0d91-5ea0-9b3e-6abc04ee57a9"
version = "7.22.0"

    [deps.ArrayInterface.extensions]
    ArrayInterfaceBandedMatricesExt = "BandedMatrices"
    ArrayInterfaceBlockBandedMatricesExt = "BlockBandedMatrices"
    ArrayInterfaceCUDAExt = "CUDA"
    ArrayInterfaceCUDSSExt = ["CUDSS", "CUDA"]
    ArrayInterfaceChainRulesCoreExt = "ChainRulesCore"
    ArrayInterfaceChainRulesExt = "ChainRules"
    ArrayInterfaceGPUArraysCoreExt = "GPUArraysCore"
    ArrayInterfaceMetalExt = "Metal"
    ArrayInterfaceReverseDiffExt = "ReverseDiff"
    ArrayInterfaceSparseArraysExt = "SparseArrays"
    ArrayInterfaceStaticArraysCoreExt = "StaticArraysCore"
    ArrayInterfaceTrackerExt = "Tracker"

    [deps.ArrayInterface.weakdeps]
    BandedMatrices = "aae01518-5342-5314-be14-df237901396f"
    BlockBandedMatrices = "ffab5731-97b5-5995-9138-79e8c1846df0"
    CUDA = "052768ef-5323-5732-b1bb-66c8b64840ba"
    CUDSS = "45b445bb-4962-46a0-9369-b4df9d0f772e"
    ChainRules = "082447d4-558c-5d27-93f4-14fc19e9eca2"
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    GPUArraysCore = "46192b85-c4d5-4398-a991-12ede77f4527"
    Metal = "dde4c033-4e86-420c-a63e-0dd931031962"
    ReverseDiff = "37e2e3b7-166d-5795-8a7a-e32c996b4267"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    StaticArraysCore = "1e83bf80-4336-4d27-bf5d-d5a4f845583c"
    Tracker = "9f7883ad-71c0-57eb-9f7f-b5c9e6d3789c"

[[deps.Artifacts]]
uuid = "56f22d72-fd6d-98f1-02f0-08ddc0907c33"

[[deps.Base64]]
uuid = "2a0f44e3-6c83-55bd-87e4-b1978d98bd5f"

[[deps.CSV]]
deps = ["CodecZlib", "Dates", "FilePathsBase", "InlineStrings", "Mmap", "Parsers", "PooledArrays", "PrecompileTools", "SentinelArrays", "Tables", "Unicode", "WeakRefStrings", "WorkerUtilities"]
git-tree-sha1 = "deddd8725e5e1cc49ee205a1964256043720a6c3"
uuid = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
version = "0.10.15"

[[deps.CodecZlib]]
deps = ["TranscodingStreams", "Zlib_jll"]
git-tree-sha1 = "962834c22b66e32aa10f7611c08c8ca4e20749a9"
uuid = "944b1d66-785c-5afd-91f1-9de20f533193"
version = "0.7.8"

[[deps.ColorSchemes]]
deps = ["ColorTypes", "ColorVectorSpace", "Colors", "FixedPointNumbers", "PrecompileTools", "Random"]
git-tree-sha1 = "b0fd3f56fa442f81e0a47815c92245acfaaa4e34"
uuid = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
version = "3.31.0"

[[deps.ColorTypes]]
deps = ["FixedPointNumbers", "Random"]
git-tree-sha1 = "67e11ee83a43eb71ddc950302c53bf33f0690dfe"
uuid = "3da002f7-5984-5a60-b8a6-cbb66c0b333f"
version = "0.12.1"

    [deps.ColorTypes.extensions]
    StyledStringsExt = "StyledStrings"

    [deps.ColorTypes.weakdeps]
    StyledStrings = "f489334b-da3d-4c2e-b8f0-e476e12c162b"

[[deps.ColorVectorSpace]]
deps = ["ColorTypes", "FixedPointNumbers", "LinearAlgebra", "Requires", "Statistics", "TensorCore"]
git-tree-sha1 = "8b3b6f87ce8f65a2b4f857528fd8d70086cd72b1"
uuid = "c3611d14-8923-5661-9e6a-0046d554d3a4"
version = "0.11.0"
weakdeps = ["SpecialFunctions"]

    [deps.ColorVectorSpace.extensions]
    SpecialFunctionsExt = "SpecialFunctions"

[[deps.Colors]]
deps = ["ColorTypes", "FixedPointNumbers", "Reexport"]
git-tree-sha1 = "37ea44092930b1811e666c3bc38065d7d87fcc74"
uuid = "5ae59095-9a9b-59fe-a467-6f913c188581"
version = "0.13.1"

[[deps.CommonMark]]
deps = ["PrecompileTools"]
git-tree-sha1 = "830073a853633d2838c52127624af3e86580a09e"
uuid = "a80b9123-70ca-4bc0-993e-6e3bcb318db6"
version = "0.10.0"
weakdeps = ["Markdown"]

    [deps.CommonMark.extensions]
    CommonMarkMarkdownExt = "Markdown"

[[deps.CommonSubexpressions]]
deps = ["MacroTools"]
git-tree-sha1 = "cda2cfaebb4be89c9084adaca7dd7333369715c5"
uuid = "bbf7d656-a473-5ed7-a52c-81e309532950"
version = "0.3.1"

[[deps.Compat]]
deps = ["TOML", "UUIDs"]
git-tree-sha1 = "9d8a54ce4b17aa5bdce0ea5c34bc5e7c340d16ad"
uuid = "34da2185-b29b-5c13-b0c7-acf172513d20"
version = "4.18.1"
weakdeps = ["Dates", "LinearAlgebra"]

    [deps.Compat.extensions]
    CompatLinearAlgebraExt = "LinearAlgebra"

[[deps.CompilerSupportLibraries_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "e66e0078-7015-5450-92f7-15fbd957f2ae"
version = "1.1.1+0"

[[deps.ConstructionBase]]
git-tree-sha1 = "b4b092499347b18a015186eae3042f72267106cb"
uuid = "187b0558-2788-49d3-abe0-74a17ed4e7c9"
version = "1.6.0"

    [deps.ConstructionBase.extensions]
    ConstructionBaseIntervalSetsExt = "IntervalSets"
    ConstructionBaseLinearAlgebraExt = "LinearAlgebra"
    ConstructionBaseStaticArraysExt = "StaticArrays"

    [deps.ConstructionBase.weakdeps]
    IntervalSets = "8197267c-284f-5f27-9208-e0e47529a953"
    LinearAlgebra = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.Crayons]]
git-tree-sha1 = "249fe38abf76d48563e2f4556bebd215aa317e15"
uuid = "a8cc5b0e-0ffa-5ad4-8c14-923d3ee1735f"
version = "4.1.1"

[[deps.DataAPI]]
git-tree-sha1 = "abe83f3a2f1b857aac70ef8b269080af17764bbe"
uuid = "9a962f9c-6df0-11e9-0e5d-c546b8b5ee8a"
version = "1.16.0"

[[deps.DataFrames]]
deps = ["Compat", "DataAPI", "DataStructures", "Future", "InlineStrings", "InvertedIndices", "IteratorInterfaceExtensions", "LinearAlgebra", "Markdown", "Missings", "PooledArrays", "PrecompileTools", "PrettyTables", "Printf", "Random", "Reexport", "SentinelArrays", "SortingAlgorithms", "Statistics", "TableTraits", "Tables", "Unicode"]
git-tree-sha1 = "d8928e9169ff76c6281f39a659f9bca3a573f24c"
uuid = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
version = "1.8.1"

[[deps.DataStructures]]
deps = ["OrderedCollections"]
git-tree-sha1 = "e357641bb3e0638d353c4b29ea0e40ea644066a6"
uuid = "864edb3b-99cc-5e75-8d2d-829cb0a9cfe8"
version = "0.19.3"

[[deps.DataValueInterfaces]]
git-tree-sha1 = "bfc1187b79289637fa0ef6d4436ebdfe6905cbd6"
uuid = "e2d170a0-9d28-54be-80f0-106bbe20a464"
version = "1.0.0"

[[deps.Dates]]
deps = ["Printf"]
uuid = "ade2ca70-3891-5945-98fb-dc099432e06a"

[[deps.DelimitedFiles]]
deps = ["Mmap"]
git-tree-sha1 = "9e2f36d3c96a820c678f2f1f1782582fcf685bae"
uuid = "8bb1440f-4735-579b-a4ab-409b98df4dab"
version = "1.9.1"

[[deps.DiffResults]]
deps = ["StaticArraysCore"]
git-tree-sha1 = "782dd5f4561f5d267313f23853baaaa4c52ea621"
uuid = "163ba53b-c6d8-5494-b064-1a9d43ac40c5"
version = "1.1.0"

[[deps.DiffRules]]
deps = ["IrrationalConstants", "LogExpFunctions", "NaNMath", "Random", "SpecialFunctions"]
git-tree-sha1 = "23163d55f885173722d1e4cf0f6110cdbaf7e272"
uuid = "b552c78f-8df3-52c6-915a-8e097449b14b"
version = "1.15.1"

[[deps.DifferentiationInterface]]
deps = ["ADTypes", "LinearAlgebra"]
git-tree-sha1 = "5e6897d988addbfe7d9ad2ee467cc0c91001aae4"
uuid = "a0c0ee7d-e4b9-4e03-894e-1c5f64a51d63"
version = "0.7.15"

    [deps.DifferentiationInterface.extensions]
    DifferentiationInterfaceChainRulesCoreExt = "ChainRulesCore"
    DifferentiationInterfaceDiffractorExt = "Diffractor"
    DifferentiationInterfaceEnzymeExt = ["EnzymeCore", "Enzyme"]
    DifferentiationInterfaceFastDifferentiationExt = "FastDifferentiation"
    DifferentiationInterfaceFiniteDiffExt = "FiniteDiff"
    DifferentiationInterfaceFiniteDifferencesExt = "FiniteDifferences"
    DifferentiationInterfaceForwardDiffExt = ["ForwardDiff", "DiffResults"]
    DifferentiationInterfaceGPUArraysCoreExt = "GPUArraysCore"
    DifferentiationInterfaceGTPSAExt = "GTPSA"
    DifferentiationInterfaceMooncakeExt = "Mooncake"
    DifferentiationInterfacePolyesterForwardDiffExt = ["PolyesterForwardDiff", "ForwardDiff", "DiffResults"]
    DifferentiationInterfaceReverseDiffExt = ["ReverseDiff", "DiffResults"]
    DifferentiationInterfaceSparseArraysExt = "SparseArrays"
    DifferentiationInterfaceSparseConnectivityTracerExt = "SparseConnectivityTracer"
    DifferentiationInterfaceSparseMatrixColoringsExt = "SparseMatrixColorings"
    DifferentiationInterfaceStaticArraysExt = "StaticArrays"
    DifferentiationInterfaceSymbolicsExt = "Symbolics"
    DifferentiationInterfaceTrackerExt = "Tracker"
    DifferentiationInterfaceZygoteExt = ["Zygote", "ForwardDiff"]

    [deps.DifferentiationInterface.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    DiffResults = "163ba53b-c6d8-5494-b064-1a9d43ac40c5"
    Diffractor = "9f5e2b26-1114-432f-b630-d3fe2085c51c"
    Enzyme = "7da242da-08ed-463a-9acd-ee780be4f1d9"
    EnzymeCore = "f151be2c-9106-41f4-ab19-57ee4f262869"
    FastDifferentiation = "eb9bf01b-bf85-4b60-bf87-ee5de06c00be"
    FiniteDiff = "6a86dc24-6348-571c-b903-95158fe2bd41"
    FiniteDifferences = "26cc04aa-876d-5657-8c51-4c34ba976000"
    ForwardDiff = "f6369f11-7733-5829-9624-2563aa707210"
    GPUArraysCore = "46192b85-c4d5-4398-a991-12ede77f4527"
    GTPSA = "b27dd330-f138-47c5-815b-40db9dd9b6e8"
    Mooncake = "da2b9cff-9c12-43a0-ae48-6db2b0edb7d6"
    PolyesterForwardDiff = "98d1487c-24ca-40b6-b7ab-df2af84e126b"
    ReverseDiff = "37e2e3b7-166d-5795-8a7a-e32c996b4267"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    SparseConnectivityTracer = "9f842d2f-2579-4b1d-911e-f412cf18a3f5"
    SparseMatrixColorings = "0a514795-09f3-496d-8182-132a7b665d35"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"
    Symbolics = "0c5d862f-8b57-4792-8d23-62f2024744c7"
    Tracker = "9f7883ad-71c0-57eb-9f7f-b5c9e6d3789c"
    Zygote = "e88e6eb3-aa80-5325-afca-941959d7151f"

[[deps.Distances]]
deps = ["LinearAlgebra", "Statistics", "StatsAPI"]
git-tree-sha1 = "c7e3a542b999843086e2f29dac96a618c105be1d"
uuid = "b4f34e82-e78d-54a5-968a-f98e89d6e8f7"
version = "0.10.12"

    [deps.Distances.extensions]
    DistancesChainRulesCoreExt = "ChainRulesCore"
    DistancesSparseArraysExt = "SparseArrays"

    [deps.Distances.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"

[[deps.Distributed]]
deps = ["Random", "Serialization", "Sockets"]
uuid = "8ba89e20-285c-5b6f-9357-94700520ee1b"

[[deps.DocStringExtensions]]
git-tree-sha1 = "7442a5dfe1ebb773c29cc2962a8980f47221d76c"
uuid = "ffbed154-4ef7-542d-bbb7-c09d3a79fcae"
version = "0.9.5"

[[deps.Downloads]]
deps = ["ArgTools", "FileWatching", "LibCURL", "NetworkOptions"]
uuid = "f43a241f-c20a-4ad4-852c-f6b1247861c6"
version = "1.6.0"

[[deps.FilePathsBase]]
deps = ["Compat", "Dates"]
git-tree-sha1 = "3bab2c5aa25e7840a4b065805c0cdfc01f3068d2"
uuid = "48062228-2e41-5def-b9a4-89aafe57970f"
version = "0.9.24"
weakdeps = ["Mmap", "Test"]

    [deps.FilePathsBase.extensions]
    FilePathsBaseMmapExt = "Mmap"
    FilePathsBaseTestExt = "Test"

[[deps.FileWatching]]
uuid = "7b1f6079-737a-58dc-b8bc-7a2ca5c1b5ee"

[[deps.FiniteDiff]]
deps = ["ArrayInterface", "LinearAlgebra", "Setfield"]
git-tree-sha1 = "9340ca07ca27093ff68418b7558ca37b05f8aeb1"
uuid = "6a86dc24-6348-571c-b903-95158fe2bd41"
version = "2.29.0"

    [deps.FiniteDiff.extensions]
    FiniteDiffBandedMatricesExt = "BandedMatrices"
    FiniteDiffBlockBandedMatricesExt = "BlockBandedMatrices"
    FiniteDiffSparseArraysExt = "SparseArrays"
    FiniteDiffStaticArraysExt = "StaticArrays"

    [deps.FiniteDiff.weakdeps]
    BandedMatrices = "aae01518-5342-5314-be14-df237901396f"
    BlockBandedMatrices = "ffab5731-97b5-5995-9138-79e8c1846df0"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.FixedPointNumbers]]
deps = ["Statistics"]
git-tree-sha1 = "05882d6995ae5c12bb5f36dd2ed3f61c98cbb172"
uuid = "53c48c17-4a7d-5ca2-90c5-79b7896eea93"
version = "0.8.5"

[[deps.Format]]
git-tree-sha1 = "9c68794ef81b08086aeb32eeaf33531668d5f5fc"
uuid = "1fa38f19-a742-5d3f-a2b9-30dd87b9d5f8"
version = "1.3.7"

[[deps.ForwardDiff]]
deps = ["CommonSubexpressions", "DiffResults", "DiffRules", "LinearAlgebra", "LogExpFunctions", "NaNMath", "Preferences", "Printf", "Random", "SpecialFunctions"]
git-tree-sha1 = "b2977f86ed76484de6f29d5b36f2fa686f085487"
uuid = "f6369f11-7733-5829-9624-2563aa707210"
version = "1.3.1"

    [deps.ForwardDiff.extensions]
    ForwardDiffStaticArraysExt = "StaticArrays"

    [deps.ForwardDiff.weakdeps]
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.Future]]
deps = ["Random"]
uuid = "9fa8497b-333b-5362-9e8d-4d0656e87820"

[[deps.Ghostscript_jll]]
deps = ["Artifacts", "JLLWrappers", "JpegTurbo_jll", "Libdl", "Zlib_jll"]
git-tree-sha1 = "38044a04637976140074d0b0621c1edf0eb531fd"
uuid = "61579ee1-b43e-5ca0-a5da-69d92c66a64b"
version = "9.55.1+0"

[[deps.HashArrayMappedTries]]
git-tree-sha1 = "2eaa69a7cab70a52b9687c8bf950a5a93ec895ae"
uuid = "076d061b-32b6-4027-95e0-9a2c6f6d7e74"
version = "0.2.0"

[[deps.Hyperscript]]
deps = ["Test"]
git-tree-sha1 = "179267cfa5e712760cd43dcae385d7ea90cc25a4"
uuid = "47d2ed2b-36de-50cf-bf87-49c2cf4b8b91"
version = "0.0.5"

[[deps.HypertextLiteral]]
deps = ["Tricks"]
git-tree-sha1 = "7134810b1afce04bbc1045ca1985fbe81ce17653"
uuid = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
version = "0.9.5"

[[deps.IOCapture]]
deps = ["Logging", "Random"]
git-tree-sha1 = "0ee181ec08df7d7c911901ea38baf16f755114dc"
uuid = "b5f81e59-6552-4d32-b1f0-c071b021bf89"
version = "1.0.0"

[[deps.InlineStrings]]
git-tree-sha1 = "8f3d257792a522b4601c24a577954b0a8cd7334d"
uuid = "842dd82b-1e85-43dc-bf29-5d0ee9dffc48"
version = "1.4.5"

    [deps.InlineStrings.extensions]
    ArrowTypesExt = "ArrowTypes"
    ParsersExt = "Parsers"

    [deps.InlineStrings.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"
    Parsers = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"

[[deps.InteractiveUtils]]
deps = ["Markdown"]
uuid = "b77e0a4c-d291-57a0-90e8-8db25a27a240"

[[deps.InvertedIndices]]
git-tree-sha1 = "6da3c4316095de0f5ee2ebd875df8721e7e0bdbe"
uuid = "41ab1584-1d38-5bbf-9106-f11c6c58b48f"
version = "1.3.1"

[[deps.IrrationalConstants]]
git-tree-sha1 = "b2d91fe939cae05960e760110b328288867b5758"
uuid = "92d709cd-6900-40b7-9082-c6be49f344b6"
version = "0.2.6"

[[deps.IteratorInterfaceExtensions]]
git-tree-sha1 = "a3f24677c21f5bbe9d2a714f95dcd58337fb2856"
uuid = "82899510-4779-5014-852e-03e436cf321d"
version = "1.0.0"

[[deps.JLLWrappers]]
deps = ["Artifacts", "Preferences"]
git-tree-sha1 = "0533e564aae234aff59ab625543145446d8b6ec2"
uuid = "692b3bcd-3c85-4b1f-b108-f13ce0eb3210"
version = "1.7.1"

[[deps.JSON]]
deps = ["Dates", "Logging", "Parsers", "PrecompileTools", "StructUtils", "UUIDs", "Unicode"]
git-tree-sha1 = "b3ad4a0255688dcb895a52fafbaae3023b588a90"
uuid = "682c06a0-de6a-54ab-a142-c8b1cf79cde6"
version = "1.4.0"

    [deps.JSON.extensions]
    JSONArrowExt = ["ArrowTypes"]

    [deps.JSON.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"

[[deps.JpegTurbo_jll]]
deps = ["Artifacts", "JLLWrappers", "Libdl"]
git-tree-sha1 = "b6893345fd6658c8e475d40155789f4860ac3b21"
uuid = "aacddb02-875f-59d6-b918-886e6ef4fbf8"
version = "3.1.4+0"

[[deps.Kaleido_jll]]
deps = ["Artifacts", "JLLWrappers", "Libdl", "Pkg"]
git-tree-sha1 = "43032da5832754f58d14a91ffbe86d5f176acda9"
uuid = "f7e6163d-2fa5-5f23-b69c-1db539e41963"
version = "0.2.1+0"

[[deps.LaTeXStrings]]
git-tree-sha1 = "dda21b8cbd6a6c40d9d02a73230f9d70fed6918c"
uuid = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
version = "1.4.0"

[[deps.Latexify]]
deps = ["Format", "Ghostscript_jll", "InteractiveUtils", "LaTeXStrings", "MacroTools", "Markdown", "OrderedCollections", "Requires"]
git-tree-sha1 = "44f93c47f9cd6c7e431f2f2091fcba8f01cd7e8f"
uuid = "23fbe1c1-3f47-55db-b15f-69d7ec21a316"
version = "0.16.10"

    [deps.Latexify.extensions]
    DataFramesExt = "DataFrames"
    SparseArraysExt = "SparseArrays"
    SymEngineExt = "SymEngine"
    TectonicExt = "tectonic_jll"

    [deps.Latexify.weakdeps]
    DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    SymEngine = "123dc426-2d89-5057-bbad-38513e3affd8"
    tectonic_jll = "d7dd28d6-a5e6-559c-9131-7eb760cdacc5"

[[deps.LibCURL]]
deps = ["LibCURL_jll", "MozillaCACerts_jll"]
uuid = "b27032c2-a3e7-50c8-80cd-2d36dbcbfd21"
version = "0.6.4"

[[deps.LibCURL_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll", "Zlib_jll", "nghttp2_jll"]
uuid = "deac9b47-8bc7-5906-a0fe-35ac56dc84c0"
version = "8.4.0+0"

[[deps.LibGit2]]
deps = ["Base64", "LibGit2_jll", "NetworkOptions", "Printf", "SHA"]
uuid = "76f85450-5226-5b5a-8eaa-529ad045b433"

[[deps.LibGit2_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll"]
uuid = "e37daf67-58a4-590a-8e99-b0245dd2ffc5"
version = "1.6.4+0"

[[deps.LibSSH2_jll]]
deps = ["Artifacts", "Libdl", "MbedTLS_jll"]
uuid = "29816b5a-b9ab-546f-933c-edad1886dfa8"
version = "1.11.0+1"

[[deps.Libdl]]
uuid = "8f399da3-3557-5675-b5ff-fb832c97cbdb"

[[deps.LineSearches]]
deps = ["LinearAlgebra", "NLSolversBase", "NaNMath", "Printf"]
git-tree-sha1 = "9ea3422d03222c6de679934d1c08f0a99405aa03"
uuid = "d3d80556-e9d4-5f37-9878-2ab0fcc64255"
version = "7.5.1"

[[deps.LinearAlgebra]]
deps = ["Libdl", "OpenBLAS_jll", "libblastrampoline_jll"]
uuid = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"

[[deps.LogExpFunctions]]
deps = ["DocStringExtensions", "IrrationalConstants", "LinearAlgebra"]
git-tree-sha1 = "13ca9e2586b89836fd20cccf56e57e2b9ae7f38f"
uuid = "2ab3a3ac-af41-5b50-aa03-7779005ae688"
version = "0.3.29"

    [deps.LogExpFunctions.extensions]
    LogExpFunctionsChainRulesCoreExt = "ChainRulesCore"
    LogExpFunctionsChangesOfVariablesExt = "ChangesOfVariables"
    LogExpFunctionsInverseFunctionsExt = "InverseFunctions"

    [deps.LogExpFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ChangesOfVariables = "9e997f8a-9a97-42d5-a9f1-ce6bfc15e2c0"
    InverseFunctions = "3587e190-3f89-42d0-90ee-14403ec27112"

[[deps.Logging]]
uuid = "56ddb016-857b-54e1-b83d-db4d58db5568"

[[deps.MIMEs]]
git-tree-sha1 = "c64d943587f7187e751162b3b84445bbbd79f691"
uuid = "6c6e2e6c-3030-632d-7369-2d6c69616d65"
version = "1.1.0"

[[deps.MacroTools]]
git-tree-sha1 = "1e0228a030642014fe5cfe68c2c0a818f9e3f522"
uuid = "1914dd2f-81c6-5fcd-8719-6d5c9610ff09"
version = "0.5.16"

[[deps.Markdown]]
deps = ["Base64"]
uuid = "d6f4376e-aef5-505a-96c1-9c027394607a"

[[deps.MbedTLS_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "c8ffd9c3-330d-5841-b78e-0817d7145fa1"
version = "2.28.2+1"

[[deps.Missings]]
deps = ["DataAPI"]
git-tree-sha1 = "ec4f7fbeab05d7747bdf98eb74d130a2a2ed298d"
uuid = "e1d29d7a-bbdc-5cf2-9ac0-f12de2c33e28"
version = "1.2.0"

[[deps.Mmap]]
uuid = "a63ad114-7e13-5084-954f-fe012c677804"

[[deps.MozillaCACerts_jll]]
uuid = "14a3606d-f60d-562e-9121-12d972cd8159"
version = "2023.1.10"

[[deps.NLSolversBase]]
deps = ["ADTypes", "DifferentiationInterface", "Distributed", "FiniteDiff", "ForwardDiff"]
git-tree-sha1 = "25a6638571a902ecfb1ae2a18fc1575f86b1d4df"
uuid = "d41bc354-129a-5804-8e4c-c37616107c6c"
version = "7.10.0"

[[deps.NLsolve]]
deps = ["Distances", "LineSearches", "LinearAlgebra", "NLSolversBase", "Printf", "Reexport"]
git-tree-sha1 = "019f12e9a1a7880459d0173c182e6a99365d7ac1"
uuid = "2774e3e8-f4cf-5e23-947b-6d7e65073b56"
version = "4.5.1"

[[deps.NaNMath]]
deps = ["OpenLibm_jll"]
git-tree-sha1 = "9b8215b1ee9e78a293f99797cd31375471b2bcae"
uuid = "77ba4419-2d1f-58cd-9bb1-8ffee604a2e3"
version = "1.1.3"

[[deps.NetworkOptions]]
uuid = "ca575930-c2e3-43a9-ace4-1e988b2c1908"
version = "1.2.0"

[[deps.OpenBLAS_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "Libdl"]
uuid = "4536629a-c528-5b80-bd46-f80d51c5b363"
version = "0.3.23+4"

[[deps.OpenLibm_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "05823500-19ac-5b8b-9628-191a04bc5112"
version = "0.8.5+0"

[[deps.OpenSpecFun_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "JLLWrappers", "Libdl"]
git-tree-sha1 = "1346c9208249809840c91b26703912dff463d335"
uuid = "efe28fd5-8261-553b-a9e1-b2916fc3738e"
version = "0.5.6+0"

[[deps.OrderedCollections]]
git-tree-sha1 = "05868e21324cede2207c6f0f466b4bfef6d5e7ee"
uuid = "bac558e1-5e72-5ebc-8fee-abe8a469f55d"
version = "1.8.1"

[[deps.Parameters]]
deps = ["OrderedCollections", "UnPack"]
git-tree-sha1 = "34c0e9ad262e5f7fc75b10a9952ca7692cfc5fbe"
uuid = "d96e819e-fc66-5662-9728-84c9c7592b0a"
version = "0.12.3"

[[deps.Parsers]]
deps = ["Dates", "PrecompileTools", "UUIDs"]
git-tree-sha1 = "7d2f8f21da5db6a806faf7b9b292296da42b2810"
uuid = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"
version = "2.8.3"

[[deps.PeriodicalDates]]
deps = ["Dates", "Printf", "RecipesBase"]
git-tree-sha1 = "e616941f8093e50a373e7d51875143213f82eca4"
uuid = "276e7ca9-e0d7-440b-97bc-a6ae82f545b1"
version = "2.0.0"

[[deps.Pkg]]
deps = ["Artifacts", "Dates", "Downloads", "FileWatching", "LibGit2", "Libdl", "Logging", "Markdown", "Printf", "REPL", "Random", "SHA", "Serialization", "TOML", "Tar", "UUIDs", "p7zip_jll"]
uuid = "44cfe95a-1eb2-52ea-b672-e2afdf69b78f"
version = "1.10.0"

[[deps.PlotlyBase]]
deps = ["ColorSchemes", "Colors", "Dates", "DelimitedFiles", "DocStringExtensions", "JSON", "LaTeXStrings", "Logging", "Parameters", "Pkg", "REPL", "Requires", "Statistics", "UUIDs"]
git-tree-sha1 = "6256ab3ee24ef079b3afa310593817e069925eeb"
uuid = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
version = "0.8.23"

    [deps.PlotlyBase.extensions]
    DataFramesExt = "DataFrames"
    DistributionsExt = "Distributions"
    IJuliaExt = "IJulia"
    JSON3Ext = "JSON3"

    [deps.PlotlyBase.weakdeps]
    DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
    Distributions = "31c24e10-a181-5473-b8eb-7969acd0382f"
    IJulia = "7073ff75-c697-5162-941a-fcdaad2a7d2a"
    JSON3 = "0f8b85d8-7281-11e9-16c2-39a750bddbf1"

[[deps.PlotlyKaleido]]
deps = ["Artifacts", "Base64", "JSON", "Kaleido_jll"]
git-tree-sha1 = "c7a270d11881c0709052f8e3def3d60d551e99f3"
uuid = "f2990250-8cf9-495f-b13a-cce12b45703c"
version = "2.3.1"

[[deps.PlutoPlotly]]
deps = ["AbstractPlutoDingetjes", "Artifacts", "ColorSchemes", "Colors", "Dates", "Downloads", "HypertextLiteral", "InteractiveUtils", "LaTeXStrings", "Markdown", "Pkg", "PlotlyBase", "PrecompileTools", "Reexport", "ScopedValues", "Scratch", "TOML"]
git-tree-sha1 = "8acd04abc9a636ef57004f4c2e6f3f6ed4611099"
uuid = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
version = "0.6.5"

    [deps.PlutoPlotly.extensions]
    PlotlyKaleidoExt = "PlotlyKaleido"
    UnitfulExt = "Unitful"

    [deps.PlutoPlotly.weakdeps]
    PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
    Unitful = "1986cc42-f94f-5a68-af5c-568840ba703d"

[[deps.PlutoTeachingTools]]
deps = ["Downloads", "HypertextLiteral", "Latexify", "Markdown", "PlutoUI"]
git-tree-sha1 = "90b41ced6bacd8c01bd05da8aed35c5458891749"
uuid = "661c6b06-c737-4d37-b85c-46df65de6f69"
version = "0.4.7"

[[deps.PlutoUI]]
deps = ["AbstractPlutoDingetjes", "Base64", "ColorTypes", "Dates", "Downloads", "FixedPointNumbers", "Hyperscript", "HypertextLiteral", "IOCapture", "InteractiveUtils", "Logging", "MIMEs", "Markdown", "Random", "Reexport", "URIs", "UUIDs"]
git-tree-sha1 = "3ac7038a98ef6977d44adeadc73cc6f596c08109"
uuid = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
version = "0.7.79"

[[deps.PooledArrays]]
deps = ["DataAPI", "Future"]
git-tree-sha1 = "36d8b4b899628fb92c2749eb488d884a926614d3"
uuid = "2dfb63ee-cc39-5dd5-95bd-886bf059d720"
version = "1.4.3"

[[deps.PrecompileTools]]
deps = ["Preferences"]
git-tree-sha1 = "5aa36f7049a63a1528fe8f7c3f2113413ffd4e1f"
uuid = "aea7be01-6a6a-4083-8856-8a6e6704d82a"
version = "1.2.1"

[[deps.Preferences]]
deps = ["TOML"]
git-tree-sha1 = "522f093a29b31a93e34eaea17ba055d850edea28"
uuid = "21216c6a-2e73-6563-6e65-726566657250"
version = "1.5.1"

[[deps.PrettyTables]]
deps = ["Crayons", "LaTeXStrings", "Markdown", "PrecompileTools", "Printf", "REPL", "Reexport", "StringManipulation", "Tables"]
git-tree-sha1 = "c5a07210bd060d6a8491b0ccdee2fa0235fc00bf"
uuid = "08abe8d2-0d0c-5749-adfa-8a2ac140af0d"
version = "3.1.2"

[[deps.Printf]]
deps = ["Unicode"]
uuid = "de0858da-6303-5e67-8744-51eddeeeb8d7"

[[deps.PtrArrays]]
git-tree-sha1 = "1d36ef11a9aaf1e8b74dacc6a731dd1de8fd493d"
uuid = "43287f4e-b6f4-7ad1-bb20-aadabca52c3d"
version = "1.3.0"

[[deps.REPL]]
deps = ["InteractiveUtils", "Markdown", "Sockets", "Unicode"]
uuid = "3fa0cd96-eef1-5676-8a61-b3b8758bbffb"

[[deps.Random]]
deps = ["SHA"]
uuid = "9a3f8284-a2c9-5f02-9a11-845980a1fd5c"

[[deps.RecipesBase]]
deps = ["PrecompileTools"]
git-tree-sha1 = "5c3d09cc4f31f5fc6af001c250bf1278733100ff"
uuid = "3cdcf5f2-1ef4-517c-9805-6587b60abb01"
version = "1.3.4"

[[deps.Reexport]]
git-tree-sha1 = "45e428421666073eab6f2da5c9d310d99bb12f9b"
uuid = "189a3867-3050-52da-a836-e630ba90ab69"
version = "1.2.2"

[[deps.Requires]]
deps = ["UUIDs"]
git-tree-sha1 = "62389eeff14780bfe55195b7204c0d8738436d64"
uuid = "ae029012-a4dd-5104-9daa-d747884805df"
version = "1.3.1"

[[deps.SHA]]
uuid = "ea8e919c-243c-51af-8825-aaa63cd721ce"
version = "0.7.0"

[[deps.ScopedValues]]
deps = ["HashArrayMappedTries", "Logging"]
git-tree-sha1 = "c3b2323466378a2ba15bea4b2f73b081e022f473"
uuid = "7e506255-f358-4e82-b7e4-beb19740aa63"
version = "1.5.0"

[[deps.Scratch]]
deps = ["Dates"]
git-tree-sha1 = "9b81b8393e50b7d4e6d0a9f14e192294d3b7c109"
uuid = "6c6a2e73-6563-6170-7368-637461726353"
version = "1.3.0"

[[deps.SentinelArrays]]
deps = ["Dates", "Random"]
git-tree-sha1 = "ebe7e59b37c400f694f52b58c93d26201387da70"
uuid = "91c51154-3ec4-41a3-a24f-3f23e20d615c"
version = "1.4.9"

[[deps.Serialization]]
uuid = "9e88b42a-f829-5b0c-bbe9-9e923198166b"

[[deps.Setfield]]
deps = ["ConstructionBase", "Future", "MacroTools", "StaticArraysCore"]
git-tree-sha1 = "c5391c6ace3bc430ca630251d02ea9687169ca68"
uuid = "efcf1570-3423-57d1-acb7-fd33fddbac46"
version = "1.1.2"

[[deps.ShiftedArrays]]
git-tree-sha1 = "503688b59397b3307443af35cd953a13e8005c16"
uuid = "1277b4bf-5013-50f5-be3d-901d8477a67a"
version = "2.0.0"

[[deps.Sockets]]
uuid = "6462fe0b-24de-5631-8697-dd941f90decc"

[[deps.SortingAlgorithms]]
deps = ["DataStructures"]
git-tree-sha1 = "64d974c2e6fdf07f8155b5b2ca2ffa9069b608d9"
uuid = "a2af1166-a08f-5f64-846c-94a0d3cef48c"
version = "1.2.2"

[[deps.SparseArrays]]
deps = ["Libdl", "LinearAlgebra", "Random", "Serialization", "SuiteSparse_jll"]
uuid = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
version = "1.10.0"

[[deps.SpecialFunctions]]
deps = ["IrrationalConstants", "LogExpFunctions", "OpenLibm_jll", "OpenSpecFun_jll"]
git-tree-sha1 = "f2685b435df2613e25fc10ad8c26dddb8640f547"
uuid = "276daf66-3868-5448-9aa4-cd146d93841b"
version = "2.6.1"

    [deps.SpecialFunctions.extensions]
    SpecialFunctionsChainRulesCoreExt = "ChainRulesCore"

    [deps.SpecialFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"

[[deps.StaticArraysCore]]
git-tree-sha1 = "6ab403037779dae8c514bad259f32a447262455a"
uuid = "1e83bf80-4336-4d27-bf5d-d5a4f845583c"
version = "1.4.4"

[[deps.Statistics]]
deps = ["LinearAlgebra", "SparseArrays"]
uuid = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
version = "1.10.0"

[[deps.StatsAPI]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "178ed29fd5b2a2cfc3bd31c13375ae925623ff36"
uuid = "82ae8749-77ed-4fe6-ae5f-f523153014b0"
version = "1.8.0"

[[deps.StatsBase]]
deps = ["AliasTables", "DataAPI", "DataStructures", "IrrationalConstants", "LinearAlgebra", "LogExpFunctions", "Missings", "Printf", "Random", "SortingAlgorithms", "SparseArrays", "Statistics", "StatsAPI"]
git-tree-sha1 = "aceda6f4e598d331548e04cc6b2124a6148138e3"
uuid = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"
version = "0.34.10"

[[deps.StringManipulation]]
deps = ["PrecompileTools"]
git-tree-sha1 = "a3c1536470bf8c5e02096ad4853606d7c8f62721"
uuid = "892a3eda-7b42-436c-8928-eab12a02cf0e"
version = "0.4.2"

[[deps.StructUtils]]
deps = ["Dates", "UUIDs"]
git-tree-sha1 = "9297459be9e338e546f5c4bedb59b3b5674da7f1"
uuid = "ec057cc2-7a8d-4b58-b3b3-92acb9f63b42"
version = "2.6.2"

    [deps.StructUtils.extensions]
    StructUtilsMeasurementsExt = ["Measurements"]
    StructUtilsTablesExt = ["Tables"]

    [deps.StructUtils.weakdeps]
    Measurements = "eff96d63-e80a-5855-80a2-b1b0885c5ab7"
    Tables = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"

[[deps.SuiteSparse_jll]]
deps = ["Artifacts", "Libdl", "libblastrampoline_jll"]
uuid = "bea87d4a-7f5b-5778-9afe-8cc45184846c"
version = "7.2.1+1"

[[deps.TOML]]
deps = ["Dates"]
uuid = "fa267f1f-6049-4f14-aa54-33bafae1ed76"
version = "1.0.3"

[[deps.TableTraits]]
deps = ["IteratorInterfaceExtensions"]
git-tree-sha1 = "c06b2f539df1c6efa794486abfb6ed2022561a39"
uuid = "3783bdb8-4a98-5b6b-af9a-565f29a5fe9c"
version = "1.0.1"

[[deps.Tables]]
deps = ["DataAPI", "DataValueInterfaces", "IteratorInterfaceExtensions", "OrderedCollections", "TableTraits"]
git-tree-sha1 = "f2c1efbc8f3a609aadf318094f8fc5204bdaf344"
uuid = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"
version = "1.12.1"

[[deps.Tar]]
deps = ["ArgTools", "SHA"]
uuid = "a4e569a6-e804-4fa4-b0f3-eef7a1d5b13e"
version = "1.10.0"

[[deps.TensorCore]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1feb45f88d133a655e001435632f019a9a1bcdb6"
uuid = "62fd8b95-f654-4bbd-a8a5-9c27f68ccd50"
version = "0.1.1"

[[deps.Test]]
deps = ["InteractiveUtils", "Logging", "Random", "Serialization"]
uuid = "8dfed614-e22c-5e08-85e1-65c5234f0b40"

[[deps.TranscodingStreams]]
git-tree-sha1 = "0c45878dcfdcfa8480052b6ab162cdd138781742"
uuid = "3bb67fe8-82b1-5028-8e26-92a6c54297fa"
version = "0.11.3"

[[deps.Tricks]]
git-tree-sha1 = "311349fd1c93a31f783f977a71e8b062a57d4101"
uuid = "410a4b4d-49e4-4fbc-ab6d-cb71b17b3775"
version = "0.1.13"

[[deps.URIs]]
git-tree-sha1 = "bef26fb046d031353ef97a82e3fdb6afe7f21b1a"
uuid = "5c2747f8-b7ea-4ff2-ba2e-563bfd36b1d4"
version = "1.6.1"

[[deps.UUIDs]]
deps = ["Random", "SHA"]
uuid = "cf7118a7-6976-5b1a-9a39-7adc72f591a4"

[[deps.UnPack]]
git-tree-sha1 = "387c1f73762231e86e0c9c5443ce3b4a0a9a0c2b"
uuid = "3a884ed6-31ef-47d7-9d2a-63182c4928ed"
version = "1.0.2"

[[deps.Unicode]]
uuid = "4ec0a83e-493e-50e2-b9ac-8f72acf5a8f5"

[[deps.WeakRefStrings]]
deps = ["DataAPI", "InlineStrings", "Parsers"]
git-tree-sha1 = "b1be2855ed9ed8eac54e5caff2afcdb442d52c23"
uuid = "ea10d353-3f73-51f8-a26c-33c1cb351aa5"
version = "1.4.2"

[[deps.WorkerUtilities]]
git-tree-sha1 = "cd1659ba0d57b71a464a29e64dbc67cfe83d54e7"
uuid = "76eceee3-57b5-4d4a-8e66-0e911cebbf60"
version = "1.6.1"

[[deps.Zlib_jll]]
deps = ["Libdl"]
uuid = "83775a58-1f1d-513f-b197-d71354ab007a"
version = "1.2.13+1"

[[deps.libblastrampoline_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850b90-86db-534c-a0d3-1478176c7d93"
version = "5.11.0+0"

[[deps.nghttp2_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850ede-7688-5339-a07c-302acd2aaf8d"
version = "1.52.0+1"

[[deps.p7zip_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "3f19e933-33d8-53b3-aaab-bd5110c3b7a0"
version = "17.4.0+2"
"""

# ╔═╡ Cell order:
# ╟─51e9ba0a-7793-4cf4-a0c1-bc90aecff9c6
# ╟─0125c893-46aa-457c-bd1b-18589e9ab2d3
# ╟─41a74197-ceb9-40b6-8d24-1712f8d93b1a
# ╟─2606832a-1d82-417a-93a8-623fe40149f6
# ╟─182cbbb9-2783-4a46-865e-036df9ee1320
# ╠═de358d50-fa53-405c-85ad-b1f0b0ce3fbd
# ╟─ded0a217-cd7b-4ba7-be70-9be19cfe8322
# ╟─69c09e47-6c6e-441c-90e2-a3863d9ca71f
# ╟─0d1f1f47-ecd8-4d16-80b4-d9154bc7cc9a
# ╟─ffadd7f9-ce62-4218-aa4d-cbc4a78b00fb
# ╟─dcfb9cc7-6816-4948-a9f9-0481974110da
# ╟─33c4fce7-c7d5-43e4-b9cf-0fac9f2b7fea
# ╟─fb354c4b-3641-4051-ac10-bd8f9559af62
# ╟─555f3004-c66f-4d69-a75a-fc5d47dc14c9
# ╟─d28dab47-c1d8-4412-8e3f-d0502fe4266a
# ╟─65fb8802-b9d5-4008-80fc-a95f0f53abdd
# ╟─d9cd05ac-b1bd-40d6-8a0d-5f50d4788b8d
# ╟─7dcfb659-f7d3-40a4-acbe-99aed0299ec7
# ╟─ad0fa392-3935-459a-9120-81ddcb47baea
# ╟─ba671d2b-cb42-4ecb-a529-5ca4fe8073be
# ╠═d7a69a24-1c35-403d-9e38-23daf35a013e
# ╟─b5b67de8-c083-4d12-adc1-ffe3023cc611
# ╟─a2c80a92-5f96-4134-9dd7-60b358f1681f
# ╟─d8c7f3dc-3343-4f68-b523-b2f037785634
# ╟─18727fde-29ed-419e-ad50-0faf1a711484
# ╟─f494ad06-2b19-47be-96be-fb6b24cb3aa0
# ╟─0f9c0bf9-c4c9-492f-8343-551a9cbadfb2
# ╟─abe31398-0b94-4783-ad8e-fa909fa3d1d3
# ╟─ff171bda-a588-4627-85bd-f0f8f913df93
# ╟─92aedf0f-c666-4372-85fe-29f28dc6fec2
# ╟─654a151d-9d09-4119-a68d-2e40c533be5a
# ╟─f1559cf3-db06-43e9-9ba4-35501496f4f0
# ╟─74010167-4771-4565-99e4-dce565e35e4b
# ╟─4b438845-97c8-4268-9a58-f97f96f6b93f
# ╟─b1f287fe-f2fd-418d-9892-d674105de650
# ╟─aa99585f-4b88-4466-9df6-d0c3d195bbf0
# ╟─a1fd4ba1-7c28-4b00-97a3-1ca2913227b0
# ╠═f0b33579-4f78-4537-a70e-073524e22f15
# ╠═8ef0f167-66cb-46a6-88d5-7be6bfec46e2
# ╠═fd300920-4528-413e-807f-c671601b3750
# ╠═5808dcf2-8efa-4bc1-93b9-e350f38f38fc
# ╠═1bdc3f5b-31fc-4abc-a75a-b7c6d0be0165
# ╟─6a3abbcf-b0bd-4e7e-a814-8977b112d27c
# ╠═82294923-9968-4a49-a94f-8ee55fea4664
# ╟─d9d34510-c614-4feb-82a3-ea5e65a6744f
# ╠═e447e90b-cf3e-40c8-8a2b-19dadf7191be
# ╟─a132c1c3-c5d9-461f-b4ac-fd356f16c928
# ╟─37c4d79d-0fbd-49e8-9166-af5f557acabf
# ╟─487fe09a-2079-407d-8cee-6e2db4fe4898
# ╠═c602906d-1116-497e-a708-2031e04014e2
# ╠═6cbcbbe1-e50c-4590-8261-dc03e4240a8f
# ╠═4824f2fe-7c60-43d8-9211-87bfa7aa4065
# ╠═b0373101-a51a-4b52-8ba5-662530a744c9
# ╟─da2bd2d5-c1a5-460b-8240-b3135aa123b1
# ╟─50de0c4e-6308-488e-b520-805dc9231349
# ╟─4970e6ab-5b01-418f-b22f-b9641ac4d53f
# ╟─a5b1d8ec-23c8-4597-84c0-0b7c93cb5d30
# ╟─ff8cd19e-8cae-47b3-8c8e-2f127779e570
# ╟─9249b41d-3e4a-4116-b9be-65636e68cac7
# ╟─8dbe4a85-3bb0-442f-8618-720c0a00af67
# ╟─e6983682-02a8-4d86-b76e-c85235931ebe
# ╟─0de12035-fc43-445e-a6f8-f97d7234de99
# ╟─fee19510-7829-4a1a-b95e-b67b0cdd6798
# ╟─ae7d434b-4623-497e-9fe4-c3b9f3f10e69
# ╟─7212caf4-6d99-4c01-92f4-c3f11be7d446
# ╟─95b13f99-c77f-41a2-ab26-8c405296fd17
# ╠═cb8fbf86-4646-4888-993d-7f602af2183f
# ╠═0b13ae26-c9ea-440d-b23c-af32664c4730
# ╠═cfb5f1f7-9e82-4b64-bede-4c93dc166c20
# ╠═0cb64be3-d306-4b2d-8ebf-020aaeb36a11
# ╠═ef67c2d4-50e5-4a75-9d1d-7f20d05ff38f
# ╠═d456b292-6c76-4484-bf76-f74490beb720
# ╠═e8b65fdc-7a03-4cb8-9865-2e168989b2a8
# ╟─37f16fc1-5450-4087-a4f4-72a03db847d4
# ╠═41d9598c-3011-4e14-8684-289a172cec10
# ╟─bd1b97c5-04ee-499b-81d7-4bf11e1a4d07
# ╟─c23a2871-6108-4c15-853a-9f2deb41ad53
# ╟─c7c5947b-4caf-4278-af25-c6efdf37557a
# ╟─c1d9b788-0454-4fb4-9eca-f08a8f693af5
# ╟─afbb7662-85e1-4a35-b82f-bce445cf0936
# ╟─4ba352ad-7f94-4f00-ae90-49c92226d0ae
# ╟─9c6874cf-6dee-489f-b9ed-89e86ecdb865
# ╟─cca6bd64-0735-4bd4-8149-7f03f7f26e57
# ╟─cffd033c-9ba9-4147-922e-2265b9e46485
# ╟─e1bcc8dd-009f-4f82-832f-0d8db11b700d
# ╟─2557c704-295a-4770-bbeb-41a94257738b
# ╟─4000f0d1-f014-4ed2-bb6e-de637e47a76d
# ╟─35c1d390-6b67-4ae4-815b-fd5eebdb1ee2
# ╟─04c46c48-a297-4955-90bd-a51b6ad9b310
# ╟─a69868ac-f4e5-4125-9584-7a0750180161
# ╟─c16f9c18-e776-45c8-bcff-66c78627928d
# ╟─1895d767-4018-43a8-94cd-174f321daca1
# ╠═05ce80f3-67e6-4e61-9447-4d703e1c67eb
# ╟─edb2c820-0b38-4f4c-8779-b0f45815b086
# ╟─ad797a54-db6f-4d85-aed2-f8f8ec774c0e
# ╠═12692d0c-2a1f-4ef2-b529-451faa7c9b00
# ╟─057e1b24-e9fa-4aa5-b9b1-97181b096851
# ╟─2c93ad8b-c489-4d37-a39f-a116715f1cfc
# ╟─5c83f904-5ff8-4056-b6a7-dc4ea8b3c993
# ╟─c644bcd3-afef-4446-ae11-2cce4b37eb88
# ╟─cb273532-f78e-48f6-9a32-5bbfc05164dc
# ╟─7f592271-13b2-4d4e-abe7-495172544e14
# ╟─8419b89c-80c1-427e-bf68-03c3a07e8ef2
# ╟─fe2181e1-ef9f-4b6a-aff2-a06f41f166df
# ╟─35edcb27-2cde-4fc8-a2ba-7f455493c627
# ╟─2b922c32-d0da-4813-b1ad-abb5167f4ce4
# ╟─1fa4dabd-3459-4187-b9f2-7e1eee56d28a
# ╟─8391202d-cec8-4b0c-bfc2-1a01419851d4
# ╟─a1cffba2-aab6-4ce5-b5c1-fe7f55f397f9
# ╟─f669ec62-dca0-42c9-a7b7-8024f6e9c7a9
# ╠═390010ce-72dd-4b69-a486-7bcc1744859f
# ╟─43a443c2-3f6d-426f-ab72-bbd67873b92e
# ╠═49f95b2d-6186-470b-956c-1e6bb3ae3de2
# ╟─abf832f5-7cb4-40e7-a27e-2bdec69c910a
# ╠═cd6c0607-ad69-455d-bc3b-6ce05fc87746
# ╟─78e2f78e-061e-4a92-8cce-4c82a46c6e31
# ╠═1a780e63-d0aa-4ad5-a76f-0314c50e3c29
# ╟─be81679d-e8b1-4dc4-b7d1-489fe363f37f
# ╠═49a6d562-d47c-470b-ae54-601fffd12c68
# ╟─be1e91a5-011d-425a-be16-ab7b45b6823f
# ╠═9362414b-e883-4569-9f31-f4ce08d9a0e5
# ╟─9c4a07a8-4144-4e01-ae25-d35c60db5951
# ╟─f84e13af-a1c3-40a5-80a6-b838410b24c7
# ╟─11a363b2-90c7-43ac-a563-257056c66908
# ╟─f1fd0927-4d06-41df-8de4-6def18e26a22
# ╟─da1d8381-5b15-475e-a693-4696689fa6d3
# ╟─fdb13094-ef0e-4626-9e57-4110c0d2ccae
# ╠═bcc72069-c9c9-4eb9-96a1-8528cd60c100
# ╠═fa1e19f1-9cbd-4454-9bf1-a1fc8cdcf52a
# ╠═5efc650a-92a5-4350-9eda-3e7b61f491f5
# ╠═ad3bb2f0-b8b7-489d-8603-53ebd01ab86f
# ╠═2bb008a1-82e2-487f-b2f5-0d27eea804de
# ╠═789cab47-ef60-4c0c-b380-42d9fadee744
# ╠═b2bc41af-aba3-484f-97ed-5558987f4d93
# ╠═531d6c8a-c325-4e9c-9496-6c69b0abed6b
# ╠═c8bdc59a-b830-41d7-903e-338a17220d18
# ╠═3a6ae0de-3774-43d1-aad0-0c1e3edd99e8
# ╟─0a09e61c-7c2b-4088-9fb8-465189ba1768
# ╠═c9b02af3-34ca-4f74-8611-0112b83942bb
# ╠═2588a536-5bfd-4af8-8b40-1a918792cd6a
# ╟─b0a9c155-65f7-44f3-a9bf-77447da741ab
# ╟─34feb3a2-01a2-48a3-87c7-47000b18c39c
# ╟─93ba50b7-098e-4b3d-a3d4-ad27a5d67e0f
# ╟─5766096f-96dc-44f9-ae72-4f75f441594f
# ╠═4e40b2ae-3e70-47bc-808f-1acd2615615a
# ╟─458274f4-636a-4d5b-893d-33f262aed1fa
# ╟─7b15c7ca-1e90-4403-8bcb-ba0194ff1950
# ╟─bdd4566f-3800-4c9c-b91e-ac3c5a65580a
# ╠═4a4cf9f2-21b5-4685-9ce8-99f76556811c
# ╟─09689df6-67a3-4f6e-aeb3-7bacc4d738e3
# ╟─79b30313-e600-4356-b589-69fbb9bdc04d
# ╟─7d10fdd7-197f-45f8-b316-c4774b0a1f79
# ╟─c306ae6c-8acc-440d-94f2-4d17c01fa999
# ╟─9c386789-44b7-4d38-95ca-beeaecb311b0
# ╟─13054251-e17d-4469-8a65-281ffd3b5999
# ╟─df7041ea-07b6-4e13-b463-b6fa7b95111c
# ╟─cac4054f-21df-4aac-be8e-ac11e30a1181
# ╟─47831542-8188-496d-b712-b8fc4db18694
# ╟─d65ae8f6-07be-48f7-a1c8-fc493c5cde43
# ╟─00000000-0000-0000-0000-000000000001
# ╟─00000000-0000-0000-0000-000000000002
